package android.rapid.connect.bluetooth;


import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothServerSocket;
import android.bluetooth.BluetoothSocket;
import android.os.Handler;
import android.rapid.connect.IDispatchListener;
import android.rapid.connect.socket.AbsThread;
import android.rapid.connect.socket.raSocketService;
import android.rapid.log.LogUtil;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.UUID;

class BluetoothSocketService extends raSocketService {
    private static final LogUtil.Tag TAG = new LogUtil.Tag("BluetoothSocketService");
    private static final String NAME_INSECURE = "BluetoothSocketService";
    // Unique UUID for this application
    private static final UUID MY_UUID_INSECURE = UUID.fromString("00001101-0000-1000-8000-00805F9B34FB");
    private final BluetoothAdapter mAdapter;

    public BluetoothSocketService(BluetoothAdapter adapter, Handler handler) {
        super(new BluetoothSocketDispatch(handler));
        mAdapter = adapter;
    }

    public synchronized void start() {
        LogUtil.d(TAG, "start");

        if (mConnectThread != null) {
            mConnectThread.cancel();
            mConnectThread = null;
        }

        if (mConnectedThread != null) {
            mConnectedThread.cancel();
            mConnectedThread = null;
        }

        setState(STATE_LISTEN);

        if (mAcceptThread == null) {
            mAcceptThread = new AcceptThread();
            mAcceptThread.start();
        }
    }

    public synchronized void connected(BluetoothSocket socket, BluetoothDevice device) {
        if (mConnectThread != null) {
            mConnectThread.cancel();
            mConnectThread = null;
        }

        if (mConnectedThread != null) {
            mConnectedThread.cancel();
            mConnectedThread = null;
        }

        if (mAcceptThread != null) {
            mAcceptThread.cancel();
            mAcceptThread = null;
        }

        mConnectedThread = new ConnectedThread(socket);
        mConnectedThread.start();

        setState(STATE_CONNECTED);
    }

    public synchronized void connect(BluetoothDevice device) {
        LogUtil.d(TAG, "connect to: " + device.getAddress());

        if (getState() == STATE_CONNECTING) {
            if (mConnectThread != null) {
                mConnectThread.cancel();
                mConnectThread = null;
            }
        }

        if (mConnectedThread != null) {
            mConnectedThread.cancel();
            mConnectedThread = null;
        }

        mConnectThread = new ConnectThread(device);
        mConnectThread.start();
        setState(STATE_CONNECTING);
    }

    public void write(byte[] out) {
        ConnectedThread r;
        synchronized (this) {
            if (getState() != STATE_CONNECTED) return;
            r = (ConnectedThread) mConnectedThread;
        }
        r.write(out);
    }

    private static class BluetoothSocketDispatch implements IDispatchListener {
        private Handler mHandler;

        public BluetoothSocketDispatch(Handler handler) {
            mHandler = handler;
        }

        @Override
        public void recv(int bytes, byte[] buffer) {
            mHandler.obtainMessage(raBluetoothConnect.MESSAGE_RECV, bytes, -1, buffer).sendToTarget();
        }

        @Override
        public void connectionFailed() {
            mHandler.obtainMessage(raBluetoothConnect.MESSAGE_CONNECT_STATE, -1, -1).sendToTarget();
        }

        @Override
        public void connectionLost() {
            mHandler.obtainMessage(raBluetoothConnect.MESSAGE_CONNECT_STATE, -2, -1).sendToTarget();
        }
    }

    private class AcceptThread extends AbsThread {
        private final LogUtil.Tag TAG = new LogUtil.Tag("AcceptThread");
        private final BluetoothServerSocket mmServerSocket;

        public AcceptThread() {
            BluetoothServerSocket tmp = null;
            try {
                tmp = mAdapter.listenUsingRfcommWithServiceRecord(
                        NAME_INSECURE, MY_UUID_INSECURE);
            } catch (IOException e) {
                LogUtil.e(TAG, "create() failed");
            }
            mmServerSocket = tmp;
            setName("AcceptThread");
        }

        @Override
        public void execute() {
            LogUtil.d(TAG, "start run()");

            BluetoothSocket socket = null;

            while (BluetoothSocketService.this.getState() != STATE_CONNECTED) {
                try {
                    socket = mmServerSocket.accept();
                } catch (IOException e) {
                    LogUtil.e(TAG, "accept() failed");
                    break;
                }

                if (socket != null) {
                    synchronized (this) {
                        switch (BluetoothSocketService.this.getState()) {
                            case STATE_LISTEN:
                            case STATE_CONNECTING:
                                connected(socket, socket.getRemoteDevice());
                                break;
                            case STATE_NONE:
                            case STATE_CONNECTED:
                                try {
                                    socket.close();
                                } catch (IOException e) {
                                    LogUtil.e(TAG, "close socket failed");
                                }
                                break;
                        }
                    }
                }
                LogUtil.i(TAG, "end run()");
            }
        }

        public void cancel() {
            LogUtil.d(TAG, "cancel()");
            try {
                mmServerSocket.close();
            } catch (IOException e) {
                LogUtil.e(TAG, "close() of server failed", e);
            }
        }
    }

    private class ConnectThread extends AbsThread {
        private final LogUtil.Tag TAG = new LogUtil.Tag("ConnectThread");
        private final BluetoothSocket mmSocket;
        private final BluetoothDevice mmDevice;

        public ConnectThread(BluetoothDevice device) {
            mmDevice = device;
            BluetoothSocket tmp = null;
            try {
                tmp = device.createRfcommSocketToServiceRecord(
                        MY_UUID_INSECURE);
            } catch (IOException e) {
                LogUtil.e(TAG, "create() failed");
            }
            mmSocket = tmp;
            setName("ConnectThread");
        }

        @Override
        public void execute() {
            LogUtil.i(TAG, "start run()");

            mAdapter.cancelDiscovery();

            try {
                mmSocket.connect();
            } catch (IOException e) {
                // Close the socket
                try {
                    mmSocket.close();
                } catch (IOException e2) {
                    LogUtil.e(TAG, "close() socket failed");
                }
                connectionFailed();
                return;
            }

            synchronized (this) {
                mConnectThread = null;
            }

            connected(mmSocket, mmDevice);
        }

        public void cancel() {
            try {
                mmSocket.close();
            } catch (IOException e) {
                LogUtil.e(TAG, "close() of connect socket failed", e);
            }
        }
    }

    private class ConnectedThread extends AbsThread {
        private final LogUtil.Tag TAG = new LogUtil.Tag("ConnectedThread");
        private final BluetoothSocket mmSocket;
        private final InputStream mmInStream;
        private final OutputStream mmOutStream;

        public ConnectedThread(BluetoothSocket socket) {
            mmSocket = socket;
            InputStream tmpIn = null;
            OutputStream tmpOut = null;

            try {
                tmpIn = socket.getInputStream();
                tmpOut = socket.getOutputStream();
            } catch (IOException e) {
                LogUtil.e(TAG, "create Input/Output Stream failed");
            }

            mmInStream = tmpIn;
            mmOutStream = tmpOut;
            setName("ConnectedThread");
        }

        @Override
        public void execute() {
            LogUtil.i(TAG, "start run()");
            byte[] buffer = new byte[1024];
            int bytes;

            while (true) {
                try {
                    bytes = mmInStream.read(buffer);
                    dispatchRecv(bytes, buffer);
                } catch (IOException e) {
                    LogUtil.e(TAG, "disconnected");
                    connectionLost();
                    BluetoothSocketService.this.start();
                    break;
                }
            }
        }

        public void write(byte[] buffer) {
            try {
                mmOutStream.write(buffer);
            } catch (IOException e) {
                LogUtil.e(TAG, "Exception during write", e);
            }
        }

        public void cancel() {
            try {
                mmSocket.close();
            } catch (IOException e) {
                LogUtil.e(TAG, "Exception during close", e);
            }
        }
    }
}
