package android.rapid.database.sqlite;

import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.rapid.log.LogUtil;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.TreeMap;

public class SQLiteHelper<T> extends SQLiteOpenHelper {
    private static final LogUtil.Tag TAG = new LogUtil.Tag("SQLiteHelper");
    private static final HashMap<Class<?>, String> CLASS_TYPES = new HashMap<Class<?>, String>();

    static {
        CLASS_TYPES.put(String.class, "TEXT");
        CLASS_TYPES.put(Integer.class, "INTEGER");
    }

    private HashMap<String, Class<?>> mColumn = new HashMap<String, Class<?>>();
    private String mTableName;
    private final Class<?> Tclass;

    public SQLiteHelper(Context context, String dbname, Class<T> t) {
        super(context, dbname, null, 1);
        Tclass = t;
        mTableName = Tclass.getSimpleName();
        registerObjectByIDBId(Tclass);
    }

    /**
     * set database table name
     *
     * @param name
     */
    public void setTableName(String name) {
        mTableName = name;
    }

    public void registerObjectByIDBId(Class<?> cls) {
        Field[] fields = cls.getDeclaredFields();
        if (fields != null && fields.length > 0) {
            for (Field field : fields) {
                try {
                    field.setAccessible(true);
                    IDBId outlet = field.getAnnotation(IDBId.class);
                    if (outlet != null) {
                        String viewId = outlet.id();
                        mColumn.put(viewId, field.getType());
                    }
                } catch (Exception e) {
                    LogUtil.e(TAG, "registerObjectByIDBId exception");
                }
            }
        }
    }

    private HashMap<String, Object> getObjectValueByIDBId(Object classObj) {
        HashMap<String, Object> columnValue = new HashMap<String, Object>();
        Field[] fields = classObj.getClass().getDeclaredFields();
        if (fields != null && fields.length > 0) {
            for (Field field : fields) {
                try {
                    field.setAccessible(true);
                    IDBId outlet = field.getAnnotation(IDBId.class);
                    if (outlet != null) {
                        String viewId = outlet.id();
                        columnValue.put(viewId, field.get(classObj));
                    }
                } catch (Exception e) {
                    LogUtil.e(TAG, "getObjectValueByIDBId exception");
                }
            }
        }
        return columnValue;
    }

    @Override
    public void onCreate(SQLiteDatabase db) {
        StringBuilder sql = new StringBuilder();
        sql.append("CREATE TABLE " + mTableName);
        sql.append(" (");
        mColumn.remove("_index");
        sql.append("_index INTEGER PRIMARY KEY AUTOINCREMENT,");
        Iterator iterator = mColumn.entrySet().iterator();
        while (iterator.hasNext()) {
            TreeMap.Entry entry = (TreeMap.Entry) iterator.next();
            sql.append(entry.getKey() + " " + CLASS_TYPES.get(entry.getValue()));
            sql.append(",");
        }
        sql.deleteCharAt(sql.length() - 1);
        sql.append(")");
        db.execSQL(sql.toString());
    }

    @Override
    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        db.execSQL("ALTER TABLE " + mTableName + " ADD COLUMN other TEXT");
    }

    public List<T> findAll() {
        ArrayList list = new ArrayList();
        HashMap<String, Object> columnValue = new HashMap<String, Object>();
        SQLiteDatabase db = getReadableDatabase();
        Cursor c = null;
        try {
            c = db.rawQuery("SELECT * FROM " + mTableName, null);
            c.moveToFirst();
            while (c.moveToNext()) {
                T obj = (T) Tclass.newInstance();
                for (String key : mColumn.keySet()) {
                    String value = c.getString(c.getColumnIndex(key));
                    columnValue.put(key, value);
                }
                Field[] fields = obj.getClass().getDeclaredFields();
                if (fields != null && fields.length > 0) {
                    for (Field field : fields) {
                        field.setAccessible(true);
                        IDBId outlet = field.getAnnotation(IDBId.class);
                        if (outlet != null) {
                            field.set(obj, columnValue.get(outlet.id()));
                        }
                    }
                }
                list.add(obj);
            }
        } catch (Exception e) {
            LogUtil.e(TAG, "findAll exception");
        } finally {
            if (c != null) {
                c.close();
            }
            db.close();
        }
        return list;
    }

    public List<T> findAll(String sql, String[] selectionArgs) {
        SQLiteDatabase db = getReadableDatabase();
        List list = new ArrayList();
        HashMap<String, Object> columnValue = new HashMap<String, Object>();
        Cursor c = null;
        try {
            c = db.rawQuery("SELECT * FROM " + mTableName + " WHERE " + sql, selectionArgs);
            while (c.moveToNext()) {
                T obj = (T) Tclass.newInstance();
                for (String key : mColumn.keySet()) {
                    String value = c.getString(c.getColumnIndex(key));
                    columnValue.put(key, value);
                }
                Field[] fields = obj.getClass().getDeclaredFields();
                if (fields != null && fields.length > 0) {
                    for (Field field : fields) {
                        field.setAccessible(true);
                        IDBId outlet = field.getAnnotation(IDBId.class);
                        if (outlet != null) {
                            field.set(obj, columnValue.get(outlet.id()));
                        }
                    }
                }
                list.add(obj);
            }
        } catch (Exception e) {
            LogUtil.e(TAG, "findAll exception");
        } finally {
            if (c != null) {
                c.close();
            }
            db.close();
        }
        return list;
    }

    public void delAll() {
        SQLiteDatabase db = getWritableDatabase();
        try {
            db.execSQL("DELETE FROM " + mTableName);
        } catch (Exception e) {
            LogUtil.e(TAG, "delAll exception");
        } finally {
            db.close();
        }
    }

    public void delAll(String sql, String[] selectionArgs) {
        SQLiteDatabase db = getWritableDatabase();
        try {
            db.execSQL("DELETE FROM " + mTableName + " WHERE " + sql, selectionArgs);
        } catch (Exception e) {
            LogUtil.e(TAG, "delAll exception");
        } finally {
            db.close();
        }
    }

    public void insert(T t) {
        StringBuilder sql_key = new StringBuilder();
        StringBuilder sql_value = new StringBuilder();
        ArrayList<Object> objects = new ArrayList<Object>();
        sql_key.append("INSERT INTO " + mTableName);
        sql_key.append(" (");
        HashMap<String, Object> columnValue = getObjectValueByIDBId(t);
        Iterator iterator = columnValue.entrySet().iterator();
        while (iterator.hasNext()) {
            TreeMap.Entry entry = (TreeMap.Entry) iterator.next();
            sql_key.append(entry.getKey());
            sql_key.append(",");
            sql_value.append("?");
            sql_value.append(",");
            objects.add(entry.getValue());
        }
        sql_key.deleteCharAt(sql_key.length() - 1);
        sql_key.append(") ");
        sql_key.append("VALUES (");
        sql_value.deleteCharAt(sql_value.length() - 1);
        sql_value.append(")");
        sql_key.append(sql_value);
        SQLiteDatabase db = getWritableDatabase();
        try {
            db.execSQL(sql_key.toString(), objects.toArray());
        } catch (Exception e) {
            LogUtil.e(TAG, "insert exception");
        } finally {
            db.close();
        }
    }

    public void update(T t) {
        StringBuilder sql_key = new StringBuilder();
        ArrayList<Object> objects = new ArrayList<Object>();
        sql_key.append("UPDATE " + mTableName + " SET ");
        HashMap<String, Object> columnValue = getObjectValueByIDBId(t);
        Object index = columnValue.get("_index");
        columnValue.remove("_index");
        Iterator iterator = columnValue.entrySet().iterator();
        while (iterator.hasNext()) {
            TreeMap.Entry entry = (TreeMap.Entry) iterator.next();
            sql_key.append(entry.getKey());
            sql_key.append("=?");
            sql_key.append(",");
            objects.add(entry.getValue());
        }
        sql_key.deleteCharAt(sql_key.length() - 1);
        sql_key.append(" WHERE _index=?");
        objects.add(index);
        SQLiteDatabase db = getWritableDatabase();
        try {
            db.execSQL(sql_key.toString(), objects.toArray());
        } catch (Exception e) {
            LogUtil.e(TAG, "update exception");
        } finally {
            db.close();
        }
    }
}