package android.rapid.database.sqlite;

import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.util.Log;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.TreeMap;

public class SQLiteHelper<T> extends SQLiteOpenHelper {
    private static final String TAG = "SQLiteHelper";
    private static final HashMap<Class<?>, String> CLASS_TYPES = new HashMap<Class<?>, String>();

    static {
        CLASS_TYPES.put(String.class, "TEXT");
        CLASS_TYPES.put(Integer.class, "INTEGER");
    }

    private HashMap<String, Class<?>> mColumn = new HashMap<String, Class<?>>();
    private HashMap<String, Object> mColumnValue = new HashMap<String, Object>();
    private String mTableName;
    private final Class<?> Tclass;

    public SQLiteHelper(Context context, String dbname) {
        super(context, dbname, null, 1);
        Tclass = getTclass(0);
        mTableName = Tclass.getSimpleName();
        registerObjectOutlet(Tclass);
    }

    private Class getTclass(int index) {
        Type genType = getClass().getGenericSuperclass();
        if (!(genType instanceof ParameterizedType)) {
            return Object.class;
        }
        Type[] params = ((ParameterizedType) genType).getActualTypeArguments();

        if (index >= params.length || index < 0) {
            throw new RuntimeException("Index outof bounds");
        }

        if (!(params[index] instanceof Class)) {
            return Object.class;
        }
        return (Class) params[index];
    }

    /**
     * set database table name
     *
     * @param name
     */
    public void setTableName(String name) {
        mTableName = name;
    }

    public void registerObjectOutlet(Class<?> cls) {
        Field[] fields = cls.getDeclaredFields();
        if (fields != null && fields.length > 0) {
            for (Field field : fields) {
                try {
                    field.setAccessible(true);
                    IDBId outlet = field.getAnnotation(IDBId.class);
                    if (outlet != null) {
                        String viewId = outlet.id();
                        mColumn.put(viewId, field.getType());
                    }
                } catch (Exception e) {
                    Log.e(TAG, "registerDBColumn exception");
                }
            }
        }
    }

    @Override
    public void onCreate(SQLiteDatabase db) {
        StringBuilder sql = new StringBuilder();
        sql.append("CREATE TABLE " + mColumnValue);
        sql.append(" (");
        mColumn.remove("_index");
        sql.append("_index INTEGER PRIMARY KEY AUTOINCREMENT,");
        Iterator iterator = mColumn.entrySet().iterator();
        while (iterator.hasNext()) {
            TreeMap.Entry entry = (TreeMap.Entry) iterator.next();
            sql.append(entry.getKey() + " " + CLASS_TYPES.get(entry.getValue()));
            sql.append(",");
        }
        sql.deleteCharAt(sql.length() - 1);
        sql.append(")");
        db.execSQL(sql.toString());
    }

    @Override
    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        db.execSQL("ALTER TABLE " + mTableName + " ADD COLUMN other TEXT");
    }

    public List<T> findAll() {
        ArrayList list = new ArrayList();
        SQLiteDatabase db = getReadableDatabase();
        Cursor c = null;
        try {
            c = db.rawQuery("SELECT * FROM " + mTableName, null);
            c.moveToFirst();
            while (c.moveToNext()) {
                T obj = (T) Tclass.newInstance();
                for (String key : mColumn.keySet()) {
                    String value = c.getString(c.getColumnIndex(key));
                    mColumnValue.put(key, value);
                }
                Field[] fields = obj.getClass().getDeclaredFields();
                if (fields != null && fields.length > 0) {
                    for (Field field : fields) {
                        field.setAccessible(true);
                        IDBId outlet = field.getAnnotation(IDBId.class);
                        if (outlet != null) {
                            field.set(obj, mColumnValue.get(outlet.id()));
                        }
                    }
                }
                list.add(obj);
            }
        } catch (Exception e) {
            Log.e(TAG, "getData exception");
        } finally {
            if (c != null) {
                c.close();
            }
        }
        return list;
    }

    public List<T> findAll(String sql, String[] selectionArgs) {
        SQLiteDatabase db = getReadableDatabase();
        List list = new ArrayList();
        try {
            Cursor c = db.rawQuery("SELECT * FROM " + mTableName + " WHERE " + sql, selectionArgs);
            while (c.moveToNext()) {
                T obj = (T) Tclass.newInstance();
                for (String key : mColumn.keySet()) {
                    String value = c.getString(c.getColumnIndex(key));
                    mColumnValue.put(key, value);
                }
                Field[] fields = obj.getClass().getDeclaredFields();
                if (fields != null && fields.length > 0) {
                    for (Field field : fields) {
                        field.setAccessible(true);
                        IDBId outlet = field.getAnnotation(IDBId.class);
                        if (outlet != null) {
                            field.set(obj, mColumnValue.get(outlet.id()));
                        }
                    }
                }
                list.add(obj);
            }
        } catch (Exception e) {
            Log.e(TAG, "findAll exception");
        } finally {
            db.close();
        }
        return list;
    }

    public void delAll() {
        SQLiteDatabase db = getWritableDatabase();
        try {
            db.execSQL("DELETE FROM " + mTableName);
        } catch (Exception e) {
            Log.e(TAG, "delAll exception");
        } finally {
            db.close();
        }
    }

    public void delAll(String sql, String[] selectionArgs) {
        SQLiteDatabase db = getWritableDatabase();
        try {
            db.execSQL("DELETE FROM " + mTableName + " WHERE " + sql, selectionArgs);
        } catch (Exception e) {
            Log.e(TAG, "delAll exception");
        } finally {
            db.close();
        }
    }
}