package cn.lsmya.helper

import android.annotation.TargetApi
import android.content.ContentUris
import android.content.Context
import android.database.Cursor
import android.net.Uri
import android.os.Build
import android.os.Environment
import android.provider.DocumentsContract
import android.provider.MediaStore
import android.provider.OpenableColumns
import androidx.annotation.NonNull
import androidx.annotation.Nullable
import java.io.*

/**
 * url转path
 */
class UriToPathHelper {
    companion object {
        private var authority = ""
        private val DOCUMENTS_DIR = "documents"

        /**
         * url转path
         */
        @JvmStatic
        fun getPath(context: Context, uri: Uri, authority: String): String {
            Companion.authority = "$authority.provider"
            val absolutePath =
                getLocalPath(
                    context,
                    uri
                )
            return absolutePath ?: uri.toString()
        }

        @TargetApi(Build.VERSION_CODES.KITKAT)
        private fun getLocalPath(context: Context, uri: Uri): String? {

            val isKitKat = Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT

            // DocumentProvider
            if (isKitKat && DocumentsContract.isDocumentUri(context, uri)) {
                // LocalStorageProvider
                if (isLocalStorageDocument(
                        uri
                    )
                ) {
                    // The path is the id
                    return DocumentsContract.getDocumentId(uri)
                } else if (isExternalStorageDocument(
                        uri
                    )
                ) {
                    val docId = DocumentsContract.getDocumentId(uri)
                    val split = docId.split(":".toRegex()).dropLastWhile { it.isEmpty() }.toTypedArray()
                    val type = split[0]

                    if ("primary".equals(type, ignoreCase = true)) {
                        return Environment.getExternalStorageDirectory().toString() + "/" + split[1]
                    } else if ("home".equals(type, ignoreCase = true)) {
                        return Environment.getExternalStorageDirectory().toString() + "/documents/" + split[1]
                    }
                } else if (isDownloadsDocument(
                        uri
                    )
                ) {

                    val id = DocumentsContract.getDocumentId(uri)

                    if (id != null && id.startsWith("raw:")) {
                        return id.substring(4)
                    }

                    val contentUriPrefixesToTry =
                        arrayOf("content://downloads/public_downloads", "content://downloads/my_downloads")

                    for (contentUriPrefix in contentUriPrefixesToTry) {
                        val contentUri =
                            ContentUris.withAppendedId(Uri.parse(contentUriPrefix), java.lang.Long.valueOf(id!!))
                        try {
                            val path =
                                getDataColumn(
                                    context,
                                    contentUri,
                                    null,
                                    null
                                )
                            if (path != null) {
                                return path
                            }
                        } catch (e: Exception) {
                        }

                    }

                    // path could not be retrieved using ContentResolver, therefore copy file to accessible cache using streams
                    val fileName =
                        getFileName(
                            context,
                            uri
                        )
                    val cacheDir =
                        getDocumentCacheDir(
                            context
                        )
                    val file =
                        generateFileName(
                            fileName,
                            cacheDir
                        )
                    var destinationPath: String? = null
                    if (file != null) {
                        destinationPath = file.absolutePath
                        saveFileFromUri(
                            context,
                            uri,
                            destinationPath
                        )
                    }

                    return destinationPath
                } else if (isMediaDocument(
                        uri
                    )
                ) {
                    val docId = DocumentsContract.getDocumentId(uri)
                    val split = docId.split(":".toRegex()).dropLastWhile { it.isEmpty() }.toTypedArray()
                    val type = split[0]

                    var contentUri: Uri? = null
                    if ("image" == type) {
                        contentUri = MediaStore.Images.Media.EXTERNAL_CONTENT_URI
                    } else if ("video" == type) {
                        contentUri = MediaStore.Video.Media.EXTERNAL_CONTENT_URI
                    } else if ("audio" == type) {
                        contentUri = MediaStore.Audio.Media.EXTERNAL_CONTENT_URI
                    }

                    val selection = "_id=?"
                    val selectionArgs = arrayOf(split[1])

                    return getDataColumn(
                        context,
                        contentUri,
                        selection,
                        selectionArgs
                    )
                }// MediaProvider
                // DownloadsProvider
                // ExternalStorageProvider
            } else if ("content".equals(uri.scheme!!, ignoreCase = true)) {

                // Return the remote address
                return if (isGooglePhotosUri(
                        uri
                    )
                ) {
                    uri.lastPathSegment
                } else getDataColumn(
                    context,
                    uri,
                    null,
                    null
                )

            } else if ("file".equals(uri.scheme!!, ignoreCase = true)) {
                return uri.path
            }// File
            // MediaStore (and general)

            return null
        }

        private fun getDataColumn(
            context: Context, uri: Uri?, selection: String?,
            selectionArgs: Array<String>?
        ): String? {

            var cursor: Cursor? = null
            val column = MediaStore.Files.FileColumns.DATA
            val projection = arrayOf(column)

            try {
                cursor = context.contentResolver.query(uri!!, projection, selection, selectionArgs, null)
                if (cursor != null && cursor.moveToFirst()) {
                    val column_index = cursor.getColumnIndexOrThrow(column)
                    return cursor.getString(column_index)
                }
            } catch (e: Exception) {
                e.printStackTrace()
            } finally {
                cursor?.close()
            }
            return null
        }

        /**
         * @param uri The Uri to check.
         * @return Whether the Uri authority is local.
         */
        private fun isLocalStorageDocument(uri: Uri): Boolean {
            return authority == uri.authority
        }

        /**
         * @param uri The Uri to check.
         * @return Whether the Uri authority is ExternalStorageProvider.
         */
        private fun isExternalStorageDocument(uri: Uri): Boolean {
            return "com.android.externalstorage.documents" == uri.authority
        }

        /**
         * @param uri The Uri to check.
         * @return Whether the Uri authority is DownloadsProvider.
         */
        private fun isDownloadsDocument(uri: Uri): Boolean {
            return "com.android.providers.downloads.documents" == uri.authority
        }

        /**
         * @param uri The Uri to check.
         * @return Whether the Uri authority is MediaProvider.
         */
        private fun isMediaDocument(uri: Uri): Boolean {
            return "com.android.providers.media.documents" == uri.authority
        }

        /**
         * @param uri The Uri to check.
         * @return Whether the Uri authority is Google Photos.
         */
        private fun isGooglePhotosUri(uri: Uri): Boolean {
            return "com.google.android.apps.photos.content" == uri.authority
        }

        private fun getFileName(@NonNull context: Context, uri: Uri): String? {
            val mimeType = context.contentResolver.getType(uri)
            var filename: String? = null

            if (mimeType == null) {
                val path = getPath(
                    context,
                    uri,
                    authority
                )
                val file = File(path)
                filename = file.name
            } else {
                val returnCursor = context.contentResolver.query(uri, null, null, null, null)
                if (returnCursor != null) {
                    val nameIndex = returnCursor.getColumnIndex(OpenableColumns.DISPLAY_NAME)
                    returnCursor.moveToFirst()
                    filename = returnCursor.getString(nameIndex)
                    returnCursor.close()
                }
            }

            return filename
        }

        private fun getDocumentCacheDir(@NonNull context: Context): File {
            val dir = File(context.cacheDir,
                DOCUMENTS_DIR
            )
            if (!dir.exists()) {
                dir.mkdirs()
            }
            return dir
        }

        @Nullable
        private fun generateFileName(@Nullable name: String?, directory: File): File? {
            var mName: String? = name ?: return null

            var file = File(directory, mName!!)

            if (file.exists()) {
                var fileName: String = mName
                var extension = ""
                val dotIndex = mName.lastIndexOf('.')
                if (dotIndex > 0) {
                    fileName = mName.substring(0, dotIndex)
                    extension = mName.substring(dotIndex)
                }

                var index = 0

                while (file.exists()) {
                    index++
                    mName = "$fileName($index)$extension"
                    file = File(directory, mName)
                }
            }

            try {
                if (!file.createNewFile()) {
                    return null
                }
            } catch (e: IOException) {
                return null
            }

            return file
        }

        private fun saveFileFromUri(context: Context, uri: Uri, destinationPath: String) {
            var inputStream: InputStream? = null
            var bos: BufferedOutputStream? = null
            try {
                inputStream = context.contentResolver.openInputStream(uri)
                bos = BufferedOutputStream(FileOutputStream(destinationPath, false))
                val buf = ByteArray(1024)
                inputStream!!.read(buf)
                do {
                    bos.write(buf)
                } while (inputStream.read(buf) != -1)
            } catch (e: IOException) {
                e.printStackTrace()
            } finally {
                try {
                    inputStream?.close()
                    bos?.close()
                } catch (e: IOException) {
                    e.printStackTrace()
                }

            }
        }

    }
}
