package cn.lsmya.helper

import android.content.Context
import android.content.SharedPreferences
import android.util.Log
import com.alibaba.fastjson.JSON

/**
 * SharedPreferences操作工具类
 */
class SpHelper {

    companion object {
        private const val defaultName: String = "default_sp_Name"
        private const val spName: String = "spNameList"

        private var mEncode = false

        /**
         * 是否打开Base64编码
         */
        @JvmStatic
        fun openEncode(encode: Boolean) {
            mEncode = encode
        }

        /**
         * 保存数据，通过apply()方式提交
         */
        @JvmStatic
        fun put(key: String, value: Any?) {
            put(
                defaultName,
                key,
                value,
                false
            )
        }

        /**
         * 保存数据到默认的sp文件中
         */
        @JvmStatic
        fun put(key: String, value: Any?, isCommit: Boolean) {
            put(
                defaultName,
                key,
                value,
                isCommit
            )
        }

        /**
         * 数据保存到自定义的sp文件中，通过apply()方式提交
         */
        @JvmStatic
        fun put(name: String, key: String, value: Any?) {
            put(
                name,
                key,
                value,
                false
            )
        }

        /**
         * 数据保存到自定义的sp文件中
         */
        @JvmStatic
        fun put(name: String, key: String, value: Any?, isCommit: Boolean) {
            if (value == null) {
                return
            }
            setSharedPreferencesList(name)
            val edit = getSharedPreferences(
                name
            ).edit()
            var editor: SharedPreferences.Editor? = null
            when (value) {
                is String -> {
                    editor = edit.putString(
                        getEncodeWord(
                            key
                        ),
                        getEncodeWord(
                            value
                        )
                    )
                }
                is Int -> {
                    editor = edit.putInt(
                        getEncodeWord(
                            key
                        ), value
                    )
                }
                is Long -> {
                    editor = edit.putLong(
                        getEncodeWord(
                            key
                        ), value
                    )
                }
                is Float -> {
                    editor = edit.putFloat(
                        getEncodeWord(
                            key
                        ), value
                    )
                }
                is Boolean -> {
                    editor = edit.putBoolean(
                        getEncodeWord(
                            key
                        ), value
                    )
                }
            }
            if (editor != null) {
                if (isCommit) {
                    editor.commit()
                } else {
                    editor.apply()
                }
            }
        }

        /**
         * 根据key在默认sp文件中获取String类型的数据
         */
        @JvmStatic
        fun getString(key: String, defaultValue: String): String {
            return getString(
                defaultName,
                key,
                defaultValue
            )
        }

        /**
         * 根据key在指定sp文件中获取String类型的数据
         */
        @JvmStatic
        fun getString(name: String, key: String, defaultValue: String): String {
            val defaultString =
                getSharedPreferences(
                    name
                )
                    .getString(
                        getEncodeWord(
                            key
                        ),
                        defaultValue
                    )
            return if (defaultString == defaultValue) {
                defaultString
            } else {
                getDecodeWord(
                    defaultString
                )
            }
        }

        /**
         * 根据key在默认sp文件中获取Int类型的数据
         */
        @JvmStatic
        fun getInt(key: String, defaultValue: Int): Int {
            return getInt(
                defaultName,
                key,
                defaultValue
            )
        }

        /**
         * 根据key在指定sp文件中获取Int类型的数据
         */
        @JvmStatic
        fun getInt(name: String, key: String, defaultValue: Int): Int {
            return getSharedPreferences(
                name
            )
                .getInt(
                    getEncodeWord(
                        key
                    ),
                    defaultValue
                )
        }

        /**
         * 根据key在默认sp文件中获取Long类型的数据
         */
        @JvmStatic
        fun getLong(key: String, defaultValue: Long?): Long? {
            return getLong(
                defaultName,
                key,
                defaultValue
            )
        }

        /**
         * 根据key在指定sp文件中获取Long类型的数据
         */
        @JvmStatic
        fun getLong(name: String, key: String, defaultValue: Long?): Long? {
            return getSharedPreferences(
                name
            ).getLong(
                getEncodeWord(
                    key
                ),
                defaultValue!!
            )
        }

        /**
         * 根据key在默认sp文件中获取Boolean类型的数据
         */
        @JvmStatic
        fun getBoolean(key: String, defaultValue: Boolean): Boolean? {
            return getBoolean(
                defaultName,
                key,
                defaultValue
            )
        }

        /**
         * 根据key在指定sp文件中获取Boolean类型的数据
         */
        @JvmStatic
        fun getBoolean(name: String, key: String, defaultValue: Boolean): Boolean? {
            return getSharedPreferences(
                getEncodeWord(
                    name
                )
            ).getBoolean(
                getEncodeWord(
                    key
                ),
                defaultValue
            )
        }

        /**
         * 根据key在默认sp文件中获取Float类型的数据
         */
        @JvmStatic
        fun getFloat(key: String, defaultValue: Float): Float? {
            return getFloat(
                defaultName,
                key,
                defaultValue
            )
        }

        /**
         * 根据key在指定sp文件中获取Float类型的数据
         */
        @JvmStatic
        fun getFloat(name: String, key: String, defaultValue: Float): Float? {
            return getSharedPreferences(
                name
            ).getFloat(
                getEncodeWord(
                    key
                ),
                defaultValue
            )
        }

        /**
         * 根据key移除默认sp文件中的数据，通过apply()方式提交
         */
        @JvmStatic
        fun removeDefault(key: String) {
            remove(
                defaultName,
                key,
                false
            )
        }

        /**
         * 根据key移除指定sp文件中的数据，通过apply()方式提交
         */
        @JvmStatic
        fun remove(name: String, key: String) {
            remove(
                name,
                key,
                false
            )
        }

        /**
         * 清除默认sp文件中的所有数据，通过apply()方式提交
         */
        @JvmStatic
        fun clearDefault() {
            clear(
                defaultName,
                false
            )
        }

        /**
         * 清除指定sp文件中的所有数据，通过apply()方式提交
         */
        @JvmStatic
        fun clear(name: String) {
            clear(
                name,
                false
            )
        }

        /**
         * 清除所有sp文件中的所有数据，通过apply()方式提交
         */
        @JvmStatic
        fun clearAll() {
            clearAll(false)
        }

        /**
         * 清除所有sp文件中的所有数据
         */
        @JvmStatic
        fun clearAll(isCommit: Boolean) {
            val list = JSON.parseArray(
                getSharedPreferences(
                    spName
                ).getString("nameList", "[]"),
                String::class.java
            )
            list?.forEach {
                clear(
                    it,
                    isCommit
                )
            }
        }

        /**
         * 根据key移除指定sp文件中的数据
         */
        @JvmStatic
        fun remove(name: String, key: String, isCommit: Boolean) {
            if (isCommit) {
                getSharedPreferences(
                    name
                ).edit().remove(
                    getEncodeWord(
                        key
                    )
                ).commit()
            } else {
                getSharedPreferences(
                    name
                ).edit().remove(
                    getEncodeWord(
                        key
                    )
                ).apply()
            }
        }

        /**
         * 根据key移除指定sp文件中的数据
         */
        @JvmStatic
        fun clear(name: String, isCommit: Boolean) {
            if (isCommit) {
                getSharedPreferences(
                    name
                ).edit().clear().commit()
            } else {
                getSharedPreferences(
                    name
                ).edit().clear().apply()
            }
        }

        private fun getSharedPreferences(name: String): SharedPreferences {
            return Tools.getContext().getSharedPreferences(name, Context.MODE_PRIVATE)
        }

        private fun getEncodeWord(value: String): String = if (mEncode) {
            Base64Helper.encodeWord(value)
        } else {
            value
        }

        private fun getDecodeWord(value: String): String = if (mEncode) {
            Base64Helper.decodeWord(value)
        } else {
            value
        }

        private fun setSharedPreferencesList(name: String) {
            val arrayList =
                getSharedPreferencesList()
            if (!arrayList.contains(name)) {
                clear(
                    spName,
                    true
                )
                arrayList.add(name)
                val jsonString = JSON.toJSONString(arrayList)
                getSharedPreferences(
                    spName
                ).edit().putString("nameList", jsonString).apply()
            }
        }

        private fun getSharedPreferencesList(): ArrayList<String> {
            val arrayList = ArrayList<String>()
            arrayList.addAll(
                JSON.parseArray(
                    getSharedPreferences(
                        spName
                    ).getString("nameList", "[]"),
                    String::class.java
                )
            )
            return arrayList
        }
    }
}