package cn.lsmya.helper;

import android.Manifest;
import android.content.Context;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.content.res.Resources;
import android.os.Build;
import android.provider.Settings;
import android.telephony.TelephonyManager;

import androidx.core.app.ActivityCompat;

import java.security.MessageDigest;
import java.util.Locale;

public class DeviceInfoHelper {

    /**
     * 获取屏幕的高
     *
     * @return 设备屏幕的高度
     */
    public static int getScreenHeight() {
        return Tools.getContext().getResources().getDisplayMetrics().heightPixels;
    }

    /**
     * 获取屏幕的宽
     *
     * @return 设备屏幕的宽度
     */
    public static int getScreenWidth() {
        return Tools.getContext().getResources().getDisplayMetrics().widthPixels;
    }

    /**
     * 得到设备的密度
     */
    public static float getScreenDensity() {
        return Tools.getContext().getResources().getDisplayMetrics().density;
    }

    /**
     * 获得状态栏的高度
     */
    public static int getStatusHeight() {
        int statusHeight = -1;
        try {
            Class<?> clazz = Class.forName("com.android.internal.R$dimen");
            Object object = clazz.newInstance();
            int id = (Integer) (clazz.getField("status_bar_height").get(object));
            statusHeight = Tools.getContext().getResources().getDimensionPixelSize(id);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return statusHeight;
    }

    public static int getStatusBarHeight(Context context) {
        Resources resources = context.getResources();
        int resourceId = resources.getIdentifier("status_bar_height", "dimen", "android");
        int height = resources.getDimensionPixelSize(resourceId);
        return height;
    }

    /**
     * 获取导航栏高度
     */
    public static int getNavigationBarHeight() {
        Resources resources = Tools.getContext().getResources();
        int identifier = resources.getIdentifier("navigation_bar_height", "dimen", "android");
        return resources.getDimensionPixelOffset(identifier);
    }

    /**
     * 获得设备序列号（如：WTK7N16923005607）, 个别设备无法获取
     *
     * @return 设备序列号
     */
    public static String getSerialNumber() {
        if (ActivityCompat.checkSelfPermission(Tools.getContext(), Manifest.permission.READ_PHONE_STATE) != PackageManager.PERMISSION_GRANTED) {
            ToastHelper.showError("请先获取读取手机设备信息权限");
            return "";
        } else {
            try {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                    return Build.getSerial();
                } else {
                    return Build.SERIAL;
                }
            } catch (Exception ex) {
                ex.printStackTrace();
            }
            return "";
        }
    }

    /**
     * 判断设备是否是手机
     *
     * @return true：是；false：否
     */
    public static boolean isPhone() {
        TelephonyManager tm = (TelephonyManager) Tools.getContext().getSystemService(Context.TELEPHONY_SERVICE);
        assert tm != null;
        return tm.getPhoneType() != TelephonyManager.PHONE_TYPE_NONE;
    }

    /**
     * 获得设备imei
     *
     * @return 设备的imei
     */
    public static String getImei() {
        if (ActivityCompat.checkSelfPermission(Tools.getContext(), Manifest.permission.READ_PHONE_STATE) != PackageManager.PERMISSION_GRANTED) {
            ToastHelper.showError("请先获取读取手机设备信息权限");
            return "";
        } else {
            try {
                TelephonyManager tm = (TelephonyManager) Tools.getContext().getSystemService(Context.TELEPHONY_SERVICE);
                assert tm != null;
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                    return tm.getImei();
                } else {
                    return tm.getDeviceId();
                }
            } catch (Exception ex) {
                ex.printStackTrace();
            }
            return "";
        }
    }

    /**
     * 获得设备的AndroidId
     *
     * @return 设备的AndroidId
     */
    public static String getAndroidId() {
        try {
            return Settings.Secure.getString(Tools.getContext().getContentResolver(), Settings.Secure.ANDROID_ID);
        } catch (Exception ex) {
            ex.printStackTrace();
        }
        return "";
    }

    /**
     * 获取设备唯一标识
     */
    public static String getDeviceId() {
        String data = getSerialNumber() + getImei() + getAndroidId();
        return bytesToHex(getHashByString(data));
    }

    /**
     * 检查当前系统是否已开启暗黑模式
     *
     * @return
     */
    public static boolean getDarkModeStatus() {
        int mode = Tools.getContext().getResources().getConfiguration().uiMode & Configuration.UI_MODE_NIGHT_MASK;
        return mode == Configuration.UI_MODE_NIGHT_YES;
    }

    /**
     * 获取hash值
     *
     * @param data
     * @return
     */
    private static byte[] getHashByString(String data) {
        try {
            MessageDigest messageDigest = MessageDigest.getInstance("SHA1");
            messageDigest.reset();
            messageDigest.update(data.getBytes("UTF-8"));
            return messageDigest.digest();
        } catch (Exception e) {
            return "".getBytes();
        }
    }

    /**
     * 转16进制字符串
     *
     * @param data 数据
     * @return 16进制字符串
     */
    private static String bytesToHex(byte[] data) {
        StringBuilder sb = new StringBuilder();
        String stmp;
        for (int n = 0; n < data.length; n++) {
            stmp = (Integer.toHexString(data[n] & 0xFF));
            if (stmp.length() == 1) {
                sb.append("0");
            }
            sb.append(stmp);
        }
        return sb.toString().toUpperCase(Locale.CHINA);
    }
}
