package cn.lsmya.helper.view;

import android.content.Context;
import android.content.res.TypedArray;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.FrameLayout;

import java.util.ArrayList;

import cn.lsmya.helper.R;

/**
 * 状态可切换的view
 *
 * @author lsm
 */
public class StatusView extends FrameLayout {

    private static final LayoutParams LAYOUT_PARAMS = new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT);

    /**
     * 空白页面
     */
    private View mEmptyView;
    /**
     * 空白页面布局id
     */
    private int mEmptyLayoutId = -1;
    /**
     * 错误页面
     */
    private View mErrorView;
    /**
     * 错误页面布局id
     */
    private int mErrorLayoutId = -1;
    /**
     * 加载页面
     */
    private View mLoadingView;
    /**
     * 加载页面布局id
     */
    private int mLoadingLayoutId = -1;
    /**
     * 内容页面
     */
    private View mContentView;
    /**
     * 内容页面布局id
     */
    private int mContentLayoutId;
    private final ArrayList<Integer> views = new ArrayList<>();

    public StatusView(Context context) {
        super(context);
        initAttrs(context, null, 0);
    }

    public StatusView(Context context, AttributeSet attrs) {
        super(context, attrs);
        initAttrs(context, attrs, 0);
    }

    public StatusView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        initAttrs(context, attrs, defStyleAttr);
    }

    private void initAttrs(Context context, AttributeSet attrs, int defStyleAttr) {
        final TypedArray typedArray = context.obtainStyledAttributes(attrs, R.styleable.StatusView, defStyleAttr, 0);
        for (int i = 0; i < typedArray.getIndexCount(); i++) {
            int attr = typedArray.getIndex(i);
            if (attr == R.styleable.StatusView_empty) {
                mEmptyLayoutId = typedArray.getResourceId(attr, -1);
            } else if (attr == R.styleable.StatusView_error) {
                mErrorLayoutId = typedArray.getResourceId(attr, -1);
            } else if (attr == R.styleable.StatusView_loading) {
                mLoadingLayoutId = typedArray.getResourceId(attr, -1);
            } else if (attr == R.styleable.StatusView_content) {
                mContentLayoutId = typedArray.getResourceId(attr, -1);
            }
        }
        typedArray.recycle();
    }

    /**
     * 显示空白视图
     */
    public void showEmpty() {
        if (mEmptyLayoutId != -1) {
            showEmpty(mEmptyLayoutId);
        }
    }

    /**
     * 显示空白视图
     *
     * @param layoutId 布局id
     */
    public void showEmpty(int layoutId) {
        if (null == mEmptyView) {
            mEmptyView = inflateView(layoutId);
            views.add(mEmptyView.getId());
            addView(mEmptyView, LAYOUT_PARAMS);
        }
        mEmptyView.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                onEmptyClick();
            }
        });
        showView(mEmptyView.getId());
    }

    /**
     * 显示错误视图
     */
    public void showError() {
        if (mErrorLayoutId != -1) {
            showError(mErrorLayoutId);
        }
    }

    /**
     * 显示错误视图
     *
     * @param layoutId 布局id
     */
    public void showError(int layoutId) {
        if (null == mErrorView) {
            mErrorView = inflateView(layoutId);
            views.add(mErrorView.getId());
            addView(mErrorView, LAYOUT_PARAMS);
        }
        mErrorView.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                onErrorClick();
            }
        });
        showView(mErrorView.getId());
    }

    /**
     * 显示加载中视图
     */
    public void showLoading() {
        if (mLoadingLayoutId != -1) {
            showLoading(mLoadingLayoutId);
        }
    }

    /**
     * 显示加载中视图
     *
     * @param layoutId 布局id
     */
    public void showLoading(int layoutId) {
        if (null == mLoadingView) {
            mLoadingView = inflateView(layoutId);
            views.add(mLoadingView.getId());
            addView(mLoadingView, LAYOUT_PARAMS);
        }
        mLoadingView.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                onLoadingClick();
            }
        });
        showView(mLoadingView.getId());
    }

    /**
     * 显示内容视图
     */
    public void showContent() {
        addContent();
        showView(mContentView.getId());
    }

    /**
     * 添加内容视图
     */
    private void addContent() {
        if (null == mContentView && mContentLayoutId != -1) {
            mContentView = inflateView(mContentLayoutId);
            views.add(mContentView.getId());
            addView(mContentView, LAYOUT_PARAMS);
        }
    }

    /**
     * 更改内容视图
     *
     * @param layoutId 布局id
     */
    public StatusView setContentView(int layoutId) {
        if (mContentView != null) {
            int id = mContentView.getId();
            int indexOf = views.indexOf(id);
            if (indexOf > -1) {
                views.remove(indexOf);
            }
        }
        mContentView = inflateView(layoutId);
        views.add(mContentView.getId());
        addView(mContentView, LAYOUT_PARAMS);
        showContent();
        return this;
    }


    /**
     * 根据id来显示view
     *
     * @param id 要显示的view的id
     */
    private void showView(int id) {
        for (int i = 0; i < getChildCount(); i++) {
            View childAt = getChildAt(i);
            childAt.setVisibility(childAt.getId() == id ? View.VISIBLE : View.GONE);
        }
    }

    /**
     * 获取布局View
     *
     * @param layoutId 布局id
     * @return 布局view
     */
    private View inflateView(int layoutId) {
        return LayoutInflater.from(getContext()).inflate(layoutId, null);
    }

    /**
     * 已添加创建完所有View
     */
    @Override
    protected void onFinishInflate() {
        super.onFinishInflate();
        addContent();
    }

    /**
     * 视图被删除
     */
    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        removeAllView(mEmptyView, mErrorView, mLoadingView, mContentView);
        views.clear();
    }

    /**
     * 空白页面点击监听
     */
    public void onEmptyClick() {

    }

    /**
     * 错误页面点击监听
     */
    public void onErrorClick() {

    }

    /**
     * 加载中页面点击监听
     */
    public void onLoadingClick() {

    }

    /**
     * 移除所有的view
     *
     * @param view 要移除的view数组
     */
    private void removeAllView(View... view) {
        for (View v : view) {
            if (v != null) {
                removeView(v);
            }
        }
    }
}
