package cn.lsmya.helper.view;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.ColorFilter;
import android.graphics.Paint;
import android.graphics.PixelFormat;
import android.graphics.drawable.Drawable;
import android.util.AttributeSet;
import android.view.Gravity;
import android.view.MotionEvent;
import android.view.ViewGroup;
import android.widget.LinearLayout;
import android.widget.ScrollView;
import android.widget.TextView;

import androidx.core.content.ContextCompat;

import java.util.List;

import cn.lsmya.helper.util.DeviceInfoHelper;

/**
 * Picker选择器
 */
public class PickerView extends ScrollView {

    /**
     * 数据源
     */
    private List<Object> mData;
    /**
     * 两条线的画笔
     */
    private Paint mLinePaint;
    /**
     * 宽,高,还有显示的TextView的高度（注意:这里只是TextView的高度,不算margin）
     */
    private int mWidth, mHeight, mTextHeight;
    /**
     * TextView的父容器
     */
    private LinearLayout mTextGroup;
    /**
     * 获取屏幕的密度
     */
    private float mDensity;
    /**
     * 字体的大小
     */
    private float mTextSize = 18;
    /**
     * 根据字体大小来控制整个控件的高度
     */
    private float mWrapContentHeight;
    /**
     * 当前选中的position
     */
    private int position, tempPosition = -1;
    private int scrollY;

    private OnSetTextListener mOnSetTextListener;
    private OnItemSelectListener mOnItemSelectListener;

    public PickerView(Context context) {
        this(context, null);
    }

    public PickerView(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public PickerView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        /**
         * 这句话设置ScrollView滑动到极限的时候不显示提示（就是那个阴影）
         */
        setOverScrollMode(OVER_SCROLL_NEVER);
        /**
         * 获取到屏幕的密度来设置TextView的高度
         */
        mDensity = getResources().getDisplayMetrics().density;
        mTextHeight = (int) ((mDensity + 0.5) * mTextSize);
        /**
         * 整个控件的高度为6个TextView的高度
         */
        mWrapContentHeight = mTextHeight * 6;
    }

    /**
     * 设置字体大小
     */
    public void setTextSize(int textSize) {
        this.mTextSize = textSize;
        mTextHeight = (int) ((mDensity + 0.5) * mTextSize);
        /**
         * 整个控件的高度为6个TextView的高度
         */
        mWrapContentHeight = mTextHeight * 6;
    }

    private void init() {

        /**
         * 初始化数据,首先添加Group
         */
        mTextGroup = new LinearLayout(getContext());
        mTextGroup.setOrientation(LinearLayout.VERTICAL);
        mTextGroup.setGravity(Gravity.CENTER);
        addView(mTextGroup);


        /**
         * 由于我们需要给自身的数据在选中的框里显示
         * 所以这里需要添加前面和后面的空数据
         */
        mTextGroup.addView(createTextView("", false));
        for (int i = 0; i < mData.size(); i++) {
            mTextGroup.addView(createTextView(mData.get(i), true));
        }
        mTextGroup.addView(createTextView("", false));

        /**
         * 设置背景,这里选择画一个,两条线
         */
        setBackground(new Drawable() {
            @Override
            public void draw(Canvas canvas) {
                /**
                 * 这里把两条线之间的距离设置为了两个TextView的高度
                 */
                canvas.drawLine(
                        mWidth * 0.1f,
                        mHeight / 2 - mTextHeight,
                        mWidth * 0.9f,
                        mHeight / 2 - mTextHeight,
                        mLinePaint
                );
                canvas.drawLine(
                        mWidth * 0.1f,
                        mHeight / 2 + mTextHeight,
                        mWidth * 0.9f,
                        mHeight / 2 + mTextHeight,
                        mLinePaint
                );
            }

            @Override
            public void setAlpha(int i) {

            }

            @Override
            public void setColorFilter(ColorFilter colorFilter) {

            }

            @Override
            public int getOpacity() {
                return PixelFormat.UNKNOWN;
            }
        });

        mLinePaint = new Paint();
        mLinePaint.setAntiAlias(true);
        mLinePaint.setStrokeWidth(1);
        mLinePaint.setColor(ContextCompat.getColor(getContext(), android.R.color.darker_gray));
        setPosition(0);

    }

    /**
     * 动态创建TextView
     */
    private TextView createTextView(Object object, boolean custom) {
        TextView tv = new TextView(getContext());
        if (custom) {
            if (mOnSetTextListener != null) {
                mOnSetTextListener.onSet(tv, object);
            }
        } else {
            tv.setText(String.valueOf(object));
        }
        if (DeviceInfoHelper.getDarkModeStatus()) {
            tv.setTextColor(Color.WHITE);
        } else {
            tv.setTextColor(ContextCompat.getColor(getContext(), android.R.color.darker_gray));
        }
        tv.setTextSize(mTextSize);
        LinearLayout.LayoutParams params = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, mTextHeight);

        params.bottomMargin = mTextHeight / 2;
        params.topMargin = mTextHeight / 2;
        tv.setLayoutParams(params);
        return tv;
    }


    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);
        /**
         * 更改测量方法
         * 指定高度为mWrapContentHeight
         */
        mWidth = measureWidth(widthMeasureSpec);
        mHeight = (int) mWrapContentHeight;
        setMeasuredDimension(mWidth, (int) mWrapContentHeight);
    }


    private int measureWidth(int size) {
        int mode = MeasureSpec.getMode(size);
        switch (mode) {
            case MeasureSpec.UNSPECIFIED:
            case MeasureSpec.AT_MOST:
                return (int) mWrapContentHeight;
            case MeasureSpec.EXACTLY:
            default:
                return MeasureSpec.getSize(size);
        }
    }

    /**
     * ScrollView的滑动事件监听
     */
    @Override
    protected void onScrollChanged(int l, int t, int oldl, int oldt) {
        super.onScrollChanged(l, t, oldl, oldt);

        /**
         * 计算出当前在两条线里的position
         */
        position = (t + mTextHeight) / (mTextHeight * 2);
        /**
         * 因为此方法会在滑动的时候不停的调用,所以这里设置一个临时的变量来控制
         */
        if (tempPosition != position) {
            int size = mTextGroup.getChildCount();
            for (int i = 0; i < size; i++) {
                TextView tv = (TextView) mTextGroup.getChildAt(i);
                /**
                 * 因为我们在数据开头添加了一个空的数据,所以这里position要+1
                 */
                if (position + 1 == i) {
                    if (DeviceInfoHelper.getDarkModeStatus()) {
                        tv.setTextColor(ContextCompat.getColor(getContext(), android.R.color.white));
                    } else {
                        tv.setTextColor(ContextCompat.getColor(getContext(), android.R.color.black));
                    }
                } else {
                    tv.setTextColor(ContextCompat.getColor(getContext(), android.R.color.darker_gray));
                }
            }
        }
        tempPosition = position;
    }

    @Override
    public boolean onTouchEvent(MotionEvent ev) {
        /**
         * 因为ScrollView没有停止滑动的监听,所以这里取巧
         * 在手指离开屏幕的30ms后判断是否和原来的scrollY一样
         * 如果一样则进入,如果不一样则设置为一样
         */
        if (ev.getAction() == MotionEvent.ACTION_UP) {
            scrollY = getScrollY();
            postDelayed(new Runnable() {
                @Override
                public void run() {
                    if (scrollY == getScrollY()) {
                        /**
                         * 获得每次松手后scrollY相对于TextView高度的偏移量
                         */
                        int offset = scrollY % (mTextHeight * 2);

                        /**
                         * 如果偏移量大于TextView高度的一半
                         * 则进入到下一个
                         */
                        if (offset > mTextHeight) {
                            smoothScrollTo(0, scrollY - offset + (mTextHeight * 2));
                        } else {
                            smoothScrollTo(0, scrollY - offset);
                        }
                    } else {
                        scrollY = getScrollY();
                        post(this);
                    }
                    if (mOnItemSelectListener != null) {
                        mOnItemSelectListener.onSelect(position);
                    }
                }
            }, 30);
        }
        return super.onTouchEvent(ev);
    }


    /**
     * 设置fling的速度为原来的1/3
     */
    @Override
    public void fling(int velocityY) {
        super.fling(velocityY / 3);
    }

    /**
     * 设置数据源
     */
    public void setData(List<Object> data, OnSetTextListener onSetTextListener) {
        mData = data;
        mOnSetTextListener = onSetTextListener;
        init();
    }

    /**
     * 设置选择item回调监听
     */
    public void setOnItemSelectListener(OnItemSelectListener listener) {
        this.mOnItemSelectListener = listener;
    }

    /**
     * 设置position
     */
    public void setPosition(int position) {
        this.position = position;
        if (position == 0) {
            post(new Runnable() {
                @Override
                public void run() {
                    scrollTo(0, 1);
                }
            });
            return;
        }
        post(new Runnable() {
            @Override
            public void run() {
                scrollTo(0, PickerView.this.position * (mTextHeight * 2));
            }
        });
    }

    /**
     * 获取position
     *
     * @return 当前选择下标
     */
    public int getPosition() {
        return position;
    }

    /**
     * 获取当前选择的内容
     *
     * @return 当前选择的内容
     */
    public Object getSelectData() {
        try {
            return mData.get(getPosition());
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 设置文本内容回调接口
     */
    public interface OnSetTextListener {
        void onSet(TextView textView, Object item);
    }

    /**
     * 手指离开屏幕时选择的item
     */
    public interface OnItemSelectListener {
        void onSelect(int position);
    }
}