package rx.library.kit.util;

import android.content.Context;
import android.content.Intent;
import android.media.MediaScannerConnection;
import android.net.Uri;
import android.os.Environment;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.DecimalFormat;

/**
 * 文件操作
 */
public class FileUtils {

    private static FileUtils fileUtils;

    public static FileUtils getInstance() {
        if (fileUtils == null) {
            fileUtils = new FileUtils();
        }
        return fileUtils;
    }

    /**
     * 获取文件类型
     *
     * @param file 文件
     * @return 文件类型
     */
    public String getFileType(File file) {
        if (!file.exists()) {
            return null;
        }
        /* 取得扩展名 */
        String end = file.getName().substring(file.getName().lastIndexOf(".") + 1).toLowerCase();
        /* 依扩展名的类型决定MimeType */
        if ("m4a".equals(end) || "mp3".equals(end) || "mid".equals(end) ||
                "xmf".equals(end) || "ogg".equals(end) || "wav".equals(end)) {
            return "audio/*";
        } else if ("3gp".equals(end) || "mp4".equals(end)) {
            return "audio/*";
        } else if ("jpg".equals(end) || "gif".equals(end) || "png".equals(end) ||
                "jpeg".equals(end) || "bmp".equals(end)) {
            return "image/*";
        } else if ("apk".equals(end)) {
            return "application/vnd.android.package-archive";
        } else if ("ppt".equals(end)) {
            return "application/vnd.ms-powerpoint";
        } else if ("xls".equals(end) || "xlsx".equals(end)) {
            return "application/vnd.ms-excel";
        } else if ("doc".equals(end) || "docx".equals(end)) {
            return "application/msword";
        } else if ("pdf".equals(end)) {
            return "application/pdf";
        } else if ("chm".equals(end)) {
            return "application/x-chm";
        } else if ("txt".equals(end)) {
            return "application/txt";
        } else {
            return "*/*";
        }
    }

    /**
     * 生成文件夹路径
     *
     * @param dir 根目录下的路径
     * @return 路径
     */
    public String getAllPath(String dir) {
        return path + "/" + dir;
    }

    private String path = Environment.getExternalStorageDirectory().toString();

    /**
     * 保存文件
     *
     * @param fileName 保存的文件名
     * @param in       文件数据
     * @throws IOException io异常
     */
    public void saveToFile(String fileName, InputStream in) throws IOException {
        FileOutputStream fos;
        BufferedInputStream bis;
        //接受1024个字节
        byte[] buf = new byte[1024];
        int size;
        bis = new BufferedInputStream(in);
        fos = new FileOutputStream(fileName);
        while ((size = bis.read(buf)) != -1) {
            fos.write(buf, 0, size);
        }
        fos.close();
        bis.close();
    }

    public File makeDir() {
        File fileDir = new File(path);
        if (fileDir.exists()) {
            return fileDir;
        } else {
            fileDir.mkdirs();
            return fileDir;
        }
    }

    /**
     * 判断文件是否存在
     *
     * @param path
     * @param fileName
     * @return
     */
    public boolean isFileExists(String path, String fileName) {
        File file = new File(path + fileName);
        boolean exists = file.exists();
        return exists;
    }

    /**
     * 删除文件夹
     *
     * @param dir 文件夹路径
     * @return
     */
    public static boolean deleteDirectory(String dir) {
        if (!dir.endsWith(File.separator)) {
            dir = dir + File.separator;
        }
        File dirFile = new File(dir);
        if ((!dirFile.exists()) || (!dirFile.isDirectory())) {
            return false;
        }
        boolean flag = true;
        File[] files = dirFile.listFiles();
        for (int i = 0; i < files.length; i++) {
            // 删除子文件
            if (files[i].isFile()) {
                flag = deleteFile(files[i].getAbsolutePath());
                if (!flag) {
                    break;
                }
            }
            // 删除子目录
            else if (files[i].isDirectory()) {
                flag = deleteDirectory(files[i]
                        .getAbsolutePath());
                if (!flag) {
                    break;
                }
            }
        }
        if (!flag) {
            return false;
        }
        // 删除当前目录
        if (dirFile.delete()) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * 删除文件
     *
     * @param filePath 要删除的文件的路径
     * @return
     */
    public static boolean deleteFile(String filePath) {
        File file = new File(filePath);
        // 如果文件路径所对应的文件存在，并且是一个文件，则直接删除
        if (file.exists() && file.isFile()) {
            if (file.delete()) {
                return true;
            } else {
                return false;
            }
        } else {
            return false;
        }
    }

    /**
     * 复制文件到SD卡
     *
     * @param context
     * @param fileName 复制的文件名
     * @param path     保存的目录路径
     * @return
     */
    public static Uri copyAssetsFile(Context context, String fileName, String path) {
        try {
            InputStream mInputStream = context.getAssets().open(fileName);
            File file = new File(path);
            if (!file.exists()) {
                file.mkdir();
            }
            File mFile = new File(path + File.separator + fileName);
            if (!mFile.exists()) {
                mFile.createNewFile();
            }
            FileOutputStream mFileOutputStream = new FileOutputStream(mFile);
            byte[] mbyte = new byte[1024];
            int i = 0;
            while ((i = mInputStream.read(mbyte)) > 0) {
                mFileOutputStream.write(mbyte, 0, i);
            }
            mInputStream.close();
            mFileOutputStream.close();
            Uri uri = RxFileProvider.getUriForFile(context, mFile);
            MediaScannerConnection.scanFile(context, new String[]{mFile.getAbsolutePath()}, null, null);
            return uri;
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 获取一个打开文件的Intent
     *
     * @param filePath 文件路径
     * @param context  上下文
     * @return 可以打开文件的Intent
     */
    public Intent openFile(String filePath, Context context) {
        File file = new File(filePath);
        if (!file.exists()) {
            return null;
        }
        /* 取得扩展名 */
        String end = file.getName().substring(file.getName().lastIndexOf(".") + 1).toLowerCase();
        /* 依扩展名的类型决定MimeType */
        if ("m4a".equals(end) || "mp3".equals(end) || "mid".equals(end) ||
                "xmf".equals(end) || "ogg".equals(end) || "wav".equals(end)) {
            return getAudioFileIntent(filePath, context);
        } else if ("3gp".equals(end) || "mp4".equals(end)) {
            return getAudioFileIntent(filePath, context);
        } else if ("jpg".equals(end) || "gif".equals(end) || "png".equals(end) ||
                "jpeg".equals(end) || "bmp".equals(end)) {
            return getImageFileIntent(filePath, context);
        } else if ("apk".equals(end)) {
            return getApkFileIntent(filePath, context);
        } else if ("ppt".equals(end)) {
            return getPptFileIntent(filePath, context);
        } else if ("xls".equals(end) || "xlsx".equals(end)) {
            return getExcelFileIntent(filePath, context);
        } else if ("doc".equals(end) || "docx".equals(end)) {
            return getWordFileIntent(filePath, context);
        } else if ("pdf".equals(end)) {
            return getPdfFileIntent(filePath, context);
        } else if ("chm".equals(end)) {
            return getChmFileIntent(filePath, context);
        } else if ("txt".equals(end)) {
            return getTextFileIntent(filePath, context);
        } else {
            return getAllIntent(filePath, context);
        }
    }

    public String getFileZHType(File file) {
        if (!file.exists()) {
            return null;
        }
        /* 取得扩展名 */
        String end = file.getName().substring(file.getName().lastIndexOf(".") + 1).toLowerCase();
        /* 依扩展名的类型决定MimeType */
        if ("m4a".equals(end) || "mp3".equals(end) || "mid".equals(end) ||
                "xmf".equals(end) || "ogg".equals(end) || "wav".equals(end)) {
            return "音乐";
        } else if ("3gp".equals(end) || "mp4".equals(end)) {
            return "视频";
        } else if ("jpg".equals(end) || "gif".equals(end) || "png".equals(end) ||
                "jpeg".equals(end) || "bmp".equals(end)) {
            return "图片";
        } else if ("apk".equals(end)) {
            return "安装包";
        } else if ("ppt".equals(end)) {
            return "PPT";
        } else if ("xls".equals(end) || "xlsx".equals(end)) {
            return "Excel";
        } else if ("doc".equals(end) || "docx".equals(end)) {
            return "Word";
        } else if ("pdf".equals(end)) {
            return "PDF";
        } else if ("chm".equals(end)) {
            return "CHM文件";
        } else if ("txt".equals(end)) {
            return "txt文本";
        } else {
            return "其他";
        }
    }

    /**
     * 转换文件大小
     *
     * @param fileS
     * @return
     */
    public String formetFileSize(long fileS) {
        DecimalFormat df = new DecimalFormat("#.00");
        String fileSizeString = "";
        String wrongSize = "0B";
        if (fileS == 0) {
            return wrongSize;
        }
        if (fileS < 1024) {
            fileSizeString = df.format((double) fileS) + "B";
        } else if (fileS < 1048576) {
            fileSizeString = df.format((double) fileS / 1024) + "KB";
        } else if (fileS < 1073741824) {
            fileSizeString = df.format((double) fileS / 1048576) + "MB";
        } else {
            fileSizeString = df.format((double) fileS / 1073741824) + "GB";
        }
        return fileSizeString;
    }

    /**
     * 获取一个用于打开文件的intent
     *
     * @param filePath 文件路径
     * @param context  上下文
     * @return 可以打开文件的Intent
     */
    private Intent getAllIntent(String filePath, Context context) {
        Intent intent = new Intent();
        intent.setFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
        Uri uri = RxFileProvider.getUriForFile(context, new File(filePath));
        intent.setAction(Intent.ACTION_VIEW);
        intent.setDataAndType(uri, "*/*");
        return intent;
    }

    /**
     * Android获取一个用于打开APK文件的intent
     *
     * @param filePath 文件路径
     * @param context  上下文
     * @return 可以打开APK文件的Intent
     */
    private Intent getApkFileIntent(String filePath, Context context) {
        Intent intent = new Intent();
        intent.setFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
        Uri uri = RxFileProvider.getUriForFile(context, new File(filePath));
        intent.setAction(Intent.ACTION_VIEW);
        intent.setDataAndType(uri, "application/vnd.android.package-archive");
        return intent;
    }

    /**
     * Android获取一个用于打开VIDEO文件的intent
     *
     * @param filePath 文件路径
     * @param context  上下文
     * @return 可以打开Video文件的Intent
     */
    private Intent getVideoFileIntent(String filePath, Context context) {
        Intent intent = new Intent("android.intent.action.VIEW");
        intent.setFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
        Uri uri = RxFileProvider.getUriForFile(context, new File(filePath));
        intent.putExtra("oneshot", 0);
        intent.putExtra("configchange", 0);
        intent.setDataAndType(uri, "video/*");
        return intent;
    }

    /**
     * Android获取一个用于打开AUDIO文件的intent
     *
     * @param filePath 文件路径
     * @param context  上下文
     * @return 可以打开Audio文件的Intent
     */
    private Intent getAudioFileIntent(String filePath, Context context) {
        Intent intent = new Intent("android.intent.action.VIEW");
        intent.setFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
        Uri uri = RxFileProvider.getUriForFile(context, new File(filePath));
        intent.putExtra("oneshot", 0);
        intent.putExtra("configchange", 0);
        intent.setDataAndType(uri, "audio/*");
        return intent;
    }

    /**
     * Android获取一个用于打开Html文件的intent
     *
     * @param filePath 文件路径
     * @param context  上下文
     * @return 可以打开Html文件的Intent
     */
    private Intent getHtmlFileIntent(String filePath, Context context) {
        Intent intent = new Intent("android.intent.action.VIEW");
        intent.setFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
        Uri uri = RxFileProvider.getUriForFile(context, new File(filePath));
        intent.setDataAndType(uri, "text/html");
        return intent;
    }

    /**
     * Android获取一个用于打开图片文件的intent
     *
     * @param filePath 文件路径
     * @param context  上下文
     * @return 可以打开图片文件的Intent
     */
    private Intent getImageFileIntent(String filePath, Context context) {
        Intent intent = new Intent("android.intent.action.VIEW");
        intent.setFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
        Uri uri = RxFileProvider.getUriForFile(context, new File(filePath));
        intent.setDataAndType(uri, "image/*");
        return intent;
    }

    /**
     * Android获取一个用于打开PPT文件的intent
     *
     * @param filePath 文件路径
     * @param context  上下文
     * @return 可以打开PPT文件的Intent
     */
    private Intent getPptFileIntent(String filePath, Context context) {
        Intent intent = new Intent("android.intent.action.VIEW");
        intent.addCategory("android.intent.category.DEFAULT");
        intent.setFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
        Uri uri = RxFileProvider.getUriForFile(context, new File(filePath));
        intent.setDataAndType(uri, "application/vnd.ms-powerpoint");
        return intent;
    }

    /**
     * Android获取一个用于打开Excel文件的intent
     *
     * @param filePath 文件路径
     * @param context  上下文
     * @return 可以打开Excel文件的Intent
     */
    private Intent getExcelFileIntent(String filePath, Context context) {
        Intent intent = new Intent("android.intent.action.VIEW");
        intent.addCategory("android.intent.category.DEFAULT");
        intent.setFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
        Uri uri = RxFileProvider.getUriForFile(context, new File(filePath));
        intent.setDataAndType(uri, "application/vnd.ms-excel");
        return intent;
    }

    /**
     * Android获取一个用于打开Word文件的intent
     *
     * @param filePath 文件路径
     * @param context  上下文
     * @return 可以打开Word文件的Intent
     */
    private Intent getWordFileIntent(String filePath, Context context) {
        Intent intent = new Intent("android.intent.action.VIEW");
        intent.addCategory("android.intent.category.DEFAULT");
        intent.setFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
        Uri uri = RxFileProvider.getUriForFile(context, new File(filePath));
        intent.setDataAndType(uri, "application/msword");
        return intent;
    }

    /**
     * Android获取一个用于打开CHM文件的intent
     *
     * @param filePath 文件路径
     * @param context  上下文
     * @return 可以打开CHM文件的Intent
     */
    private Intent getChmFileIntent(String filePath, Context context) {
        Intent intent = new Intent("android.intent.action.VIEW");
        intent.addCategory("android.intent.category.DEFAULT");
        intent.setFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
        Uri uri = RxFileProvider.getUriForFile(context, new File(filePath));
        intent.setDataAndType(uri, "application/x-chm");
        return intent;
    }

    /**
     * Android获取一个用于打开文本文件的intent
     *
     * @param filePath     文件路径
     * @param paramBoolean
     * @return 可以打开文本文件的Intent
     */
    private Intent getTextFileIntent(String filePath, boolean paramBoolean) {
        Intent intent = new Intent("android.intent.action.VIEW");
        intent.addCategory("android.intent.category.DEFAULT");
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        if (paramBoolean) {
            Uri uri1 = Uri.parse(filePath);
            intent.setDataAndType(uri1, "text/plain");
        } else {
            Uri uri2 = Uri.fromFile(new File(filePath));
            intent.setDataAndType(uri2, "text/plain");
        }
        return intent;
    }

    /**
     * Android获取一个用于打开文本文件的intent
     *
     * @param filePath 文件路径
     * @param context  上下文
     * @return 可以打开文本文件的Intent
     */
    private Intent getTextFileIntent(String filePath, Context context) {
        Intent intent = new Intent("android.intent.action.VIEW");
        intent.addCategory("android.intent.category.DEFAULT");
        intent.setFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
        Uri uri = RxFileProvider.getUriForFile(context, new File(filePath));
        intent.setDataAndType(uri, "text/plain");
        return intent;
    }

    /**
     * Android获取一个用于打开PDF文件的intent
     *
     * @param filePath 文件路径
     * @param context  上下文
     * @return 可以打开PDF文件的Intent
     */
    private Intent getPdfFileIntent(String filePath, Context context) {
        Intent intent = new Intent("android.intent.action.VIEW");
        intent.addCategory("android.intent.category.DEFAULT");
        intent.setFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
        Uri uri = RxFileProvider.getUriForFile(context, new File(filePath));
        intent.setDataAndType(uri, "application/pdf");
        return intent;
    }


}
