package rx.library.kit.activity

import android.app.Dialog
import android.graphics.PorterDuff
import android.os.Build
import android.os.Bundle
import android.text.Spannable
import android.text.SpannableStringBuilder
import android.text.TextUtils
import android.text.style.ForegroundColorSpan
import android.view.*
import android.widget.RelativeLayout
import androidx.annotation.DrawableRes
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.widget.Toolbar
import com.google.android.material.appbar.AppBarLayout
import com.gyf.immersionbar.ImmersionBar
import rx.library.kit.R
import rx.library.kit.RxKit
import rx.library.kit.RxKit.Companion.getContext
import rx.library.kit.loading.LoadingView
import rx.library.kit.util.DensityUtils

open class BaseToolBarActivity : AppCompatActivity() {

    private lateinit var toolbar: Toolbar
    private lateinit var appBarLayout: AppBarLayout
    private lateinit var contentLayoutView: RelativeLayout
    private var loadingView: View? = null
    private var emptyView: View? = null
    private var loadingDialog: Dialog? = null

    override fun setContentView(layoutResID: Int) {
        LayoutInflater.from(this).inflate(layoutResID, contentLayoutView)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        LayoutInflater.from(this).inflate(
            R.layout.kit_toolbar_layout,
            window.decorView.findViewById(android.R.id.content)
        )
        findView()
        initToolbar()


    }

    private fun findView() {
        contentLayoutView = findViewById(R.id.kit_toolbar_layout_content)
        toolbar = findViewById(R.id.kit_toolbar_layout_toolbar)
        appBarLayout = findViewById(R.id.kit_toolbar_layout_appBar)
    }

    private fun initToolbar() {
        getAppBarLayout().setPadding(0, getStatusHeight(), 0, 0)
        getAppBarLayout().setBackgroundColor(RxKit.getTitleBackground())
        getToolbar().title = getTitleText()
        getToolbar().setTitleTextColor(RxKit.getTitleTextColor())
        if (showBackButton()) {
            getToolbar().navigationIcon = RxKit.getTitleBackIcon()
        }
        getToolbar().contentInsetStartWithNavigation = 0
        getToolbar().setNavigationOnClickListener { setNavigationIconClick() }
        getToolbar().setBackgroundColor(RxKit.getTitleBackground())
        if (showToolbar()) {
            getAppBarLayout().visibility = View.VISIBLE
        } else {
            getAppBarLayout().visibility = View.GONE
        }
        setStatusBarMode(RxKit.getStatusBarMode())

    }

    fun hideAppBarLayoutElevation() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            getAppBarLayout().elevation = 0f
        }
    }

    /**
     * 是否显示toolbar
     */
    open fun showToolbar(): Boolean {
        return true
    }

    fun setToolbarHide() {
        getAppBarLayout().visibility = View.GONE
    }

    fun setToolbarShow() {
        getAppBarLayout().visibility = View.VISIBLE
    }

    fun getAppBarLayout() = appBarLayout

    fun getToolbar() = toolbar

    fun setToolbarBackgroundColor(color: Int) {
        getAppBarLayout().setBackgroundColor(color)
        getToolbar().setBackgroundColor(color)
    }

    fun setToolbarTitleTextColor(color: Int) {
        getToolbar().setTitleTextColor(color)
        for (i in 0 until getMenuCount()) {
            val menuItem = getMenuItem(i)
            menuItem.icon?.setColorFilter(RxKit.getTitleTextColor(), PorterDuff.Mode.SRC_ATOP)
            val titleStr = menuItem.title
            if (!TextUtils.isEmpty(titleStr)) {
                menuItem.title = addColor(titleStr, RxKit.getTitleTextColor())
            }
        }
        getToolbar().navigationIcon?.setColorFilter(
            RxKit.getTitleTextColor(),
            PorterDuff.Mode.SRC_ATOP
        )
    }

    /**
     * 控制是否显示导航栏左侧按钮
     */
    open fun showBackButton() = true

    /**
     * 返回上一层
     */
    open fun setNavigationIconClick() {
        finish()
    }

    /**
     * 获取标题内容，可重新自定义设置标题文本内容
     */
    open fun getTitleText(): String {
        return title.toString()
    }

    /**
     * 设置标题内容
     */
    fun setTitleText(titleStr: String) {
        getToolbar().title = titleStr
    }

    fun setNavigationIcon(iconId: Int) {
        getToolbar().setNavigationIcon(iconId)
    }

    fun getNavigationIcon() = getToolbar().navigationIcon

    fun showLoadingDialog(show: Boolean) {
        if (loadingDialog == null) {
            loadingDialog = Dialog(this)
        }
        loadingDialog!!.window.setBackgroundDrawableResource(android.R.color.transparent)
        loadingDialog!!.window.setDimAmount(0f)
        loadingDialog!!.setCanceledOnTouchOutside(false)
        val loadingView = LoadingView(this)
        loadingView.setColor(RxKit.getLoadingDialogColor())
        loadingView.setSize(DensityUtils.dp2px(RxKit.getLoadingDialogSize()))
        loadingDialog!!.setContentView(loadingView)
        if (show) {
            if (!loadingDialog!!.isShowing) {
                loadingDialog!!.show()
            }
        } else {
            if (loadingDialog!!.isShowing) {
                loadingDialog!!.dismiss()
            }
        }
    }

    /**
     * 添加一个文字菜单按钮
     */
    fun addMenuItemTextView(text: String, onClickListener: MenuItem.OnMenuItemClickListener) {
        val menuItem = getToolbar().menu.add(addColor(text, RxKit.getTitleTextColor()))
        menuItem.setOnMenuItemClickListener(onClickListener)
        menuItem.setShowAsAction(MenuItem.SHOW_AS_ACTION_ALWAYS)
    }

    /**
     * 修改菜单按钮的文字内容
     */
    fun changeMenuItemTextView(text: String, item: MenuItem) {
        changeMenuItemTextView(text, item, RxKit.getTitleTextColor())
    }

    fun changeMenuItemTextView(text: String, item: MenuItem, color: Int) {
        item.title = addColor(text, color)
    }

    override fun onPrepareOptionsMenu(menu: Menu?): Boolean {
        return super.onPrepareOptionsMenu(menu)
    }

    /**
     * 添加一个图标菜单按钮
     */
    fun addMenuItemImageView(@DrawableRes resId: Int, onClickListener: MenuItem.OnMenuItemClickListener) {
        val menuItem = getToolbar().menu.add("")
        menuItem.setIcon(resId)
        menuItem.setOnMenuItemClickListener(onClickListener)
        menuItem.setShowAsAction(MenuItem.SHOW_AS_ACTION_ALWAYS)
        menuItem.icon.setColorFilter(RxKit.getTitleTextColor(), PorterDuff.Mode.SRC_ATOP)
    }


    override fun onResume() {
        super.onResume()
        for (i in 0 until getMenuCount()) {
            val menuItem = getMenuItem(i)
            menuItem.icon?.setColorFilter(RxKit.getTitleTextColor(), PorterDuff.Mode.SRC_ATOP)
            val titleStr = menuItem.title
            if (!TextUtils.isEmpty(titleStr)) {
                menuItem.title = addColor(titleStr, RxKit.getTitleTextColor())
            }
        }
        getToolbar().navigationIcon?.setColorFilter(
            RxKit.getTitleTextColor(),
            PorterDuff.Mode.SRC_ATOP
        )
    }

    fun getMenuItem(position: Int): MenuItem {
        return getToolbar().menu.getItem(position)
    }

    fun getMenuCount() = getToolbar().menu.size()

    private fun addColor(text: CharSequence, color: Int): SpannableStringBuilder {
        val builder = SpannableStringBuilder(text)
        if (color != 0) {
            builder.setSpan(
                ForegroundColorSpan(color), 0, text.length,
                Spannable.SPAN_EXCLUSIVE_EXCLUSIVE
            )
        }
        return builder
    }

    /**
     * 设置状态栏黑色字体和图标
     */
    fun setStatusBarLightMode() {
        if (ImmersionBar.isSupportStatusBarDarkFont()) {
            ImmersionBar.with(this)
                .statusBarDarkFont(true)
                .init()
        }
    }

    /**
     * 设置状态栏白色字体和图标
     */
    fun setStatusBarDarkMode() {
        ImmersionBar.with(this)
            .statusBarDarkFont(false)
            .init()
    }

    fun setStatusBarMode(lightMode: Boolean) {
        if (lightMode) {
            setStatusBarLightMode()
        } else {
            setStatusBarDarkMode()
        }
    }

    /**
     * 显示加载布局
     */
    fun showLoading(show: Boolean) {
        if (show) {
            if (loadingView == null) {
                loadingView = LayoutInflater.from(this).inflate(RxKit.getLoadingLayout(), null)
                getRootViewGroup().addView(loadingView)
            }
        } else {
            getRootViewGroup().removeView(loadingView)
            loadingView = null
        }
    }

    /**
     * 显示空布局
     */
    fun showEmpty(show: Boolean) {
        if (show) {
            if (emptyView == null) {
                emptyView = LayoutInflater.from(this).inflate(RxKit.getEmptyLayout(), null)
                getRootViewGroup().addView(emptyView)
            }
        } else {
            getRootViewGroup().removeView(emptyView)
            emptyView = null
        }
    }

    private fun getRootViewGroup(): ViewGroup {
        return getRootView() as ViewGroup
    }

    private fun getRootView(): View {
        return window.decorView.findViewById(android.R.id.content)
    }

    override fun onOptionsItemSelected(item: MenuItem?): Boolean {
        val itemId = item!!.itemId
        if (itemId == android.R.id.home) {
            setNavigationIconClick()
        }
        return super.onOptionsItemSelected(item)
    }

    override fun onMenuOpened(featureId: Int, menu: Menu?): Boolean {
        if (menu != null) {
            if (menu.javaClass.simpleName == "MenuBuilder") {
                try {
                    val method = menu.javaClass
                        .getDeclaredMethod("setOptionalIconsVisible", java.lang.Boolean.TYPE)
                    method.isAccessible = true
                    method.invoke(menu, true)
                } catch (e: Exception) {
                    e.printStackTrace()
                }

            }
        }
        return super.onMenuOpened(featureId, menu)
    }

    private fun getStatusHeight(): Int {
        var statusHeight = -1
        try {
            val clazz = Class.forName("com.android.internal.R\$dimen")
            val `object` = clazz.newInstance()
            val id = clazz.getField("status_bar_height")[`object`] as Int
            statusHeight = getContext().resources.getDimensionPixelSize(id)
        } catch (e: java.lang.Exception) {
            e.printStackTrace()
        }
        return statusHeight
    }

}