package rx.library.kit.activity

import android.app.Activity
import android.app.Dialog
import android.graphics.PorterDuff
import android.graphics.drawable.ColorDrawable
import android.text.Spannable
import android.text.SpannableStringBuilder
import android.text.TextUtils
import android.text.style.ForegroundColorSpan
import android.view.*
import android.view.LayoutInflater
import android.view.inputmethod.InputMethodManager
import androidx.annotation.MenuRes
import androidx.appcompat.app.ActionBar
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.widget.Toolbar
import com.gyf.immersionbar.ImmersionBar
import rx.library.kit.R
import rx.library.kit.RxKit
import rx.library.kit.loading.LoadingView
import rx.library.kit.util.DensityUtils

open class BaseToolBarMenuActivity : AppCompatActivity() {

    private lateinit var toolbar: Toolbar
    private var loadingView: View? = null
    private var emptyView: View? = null
    private var loadingDialog: Dialog? = null

    override fun setContentView(view: View?) {
        super.setContentView(view)
        setCustomActionBar()
    }

    override fun setContentView(view: View?, params: ViewGroup.LayoutParams?) {
        super.setContentView(view, params)
        setCustomActionBar()
    }

    override fun setContentView(layoutResID: Int) {
        setCustomActionBar()

        super.setContentView(layoutResID)
    }

    private fun setCustomActionBar() {
        val lp = ActionBar.LayoutParams(
            ActionBar.LayoutParams.MATCH_PARENT,
            ActionBar.LayoutParams.MATCH_PARENT,
            Gravity.CENTER
        )
        val mActionBarView =
            LayoutInflater.from(this).inflate(R.layout.kit_toolbar, null)
        toolbar = mActionBarView.findViewById(R.id.kit_toolbar)

        toolbar.title = getTitleText()
        toolbar.setTitleTextColor(getTitleTextColor())
        if (showBackButton()) {
            toolbar.navigationIcon = RxKit.getTitleBackIcon()
        }
        toolbar.contentInsetStartWithNavigation = 0
        toolbar.setNavigationOnClickListener { setNavigationIconClick() }
        toolbar.setBackgroundColor(RxKit.getTitleBackground())
        val actionBar = supportActionBar
        actionBar!!.setBackgroundDrawable(ColorDrawable(RxKit.getTitleBackground()))
        actionBar.setCustomView(mActionBarView, lp)
        actionBar.displayOptions = ActionBar.DISPLAY_SHOW_CUSTOM

        setStatusBarMode(RxKit.getStatusBarMode())
    }

    fun getToolbar() = toolbar

    fun setToolbarBackgroundColor(color: Int) {
        getToolbar().setBackgroundColor(color)
        supportActionBar!!.setBackgroundDrawable(ColorDrawable(color))
    }

    fun setToolbarTitleTextColor(color: Int) {
        getToolbar().setTitleTextColor(color)
    }

    /**
     * @return menu
     */
    @MenuRes
    protected open fun menu(): Int {
        return 0
    }

    override fun onCreateOptionsMenu(menu: Menu?): Boolean {
        if (menu() != 0) {
            menuInflater.inflate(menu(), menu)
        }
        onMenuCreate(menu)
        return super.onCreateOptionsMenu(menu)
    }

    protected open fun onMenuCreate(menu: Menu?) {}

    /**
     * 控制是否显示导航栏左侧按钮
     */
    open fun showBackButton() = true

    /**
     * 返回上一层
     */
    open fun setNavigationIconClick() {
        finish()
    }

    /**
     * 获取标题内容，可重新自定义设置标题文本内容
     */
    open fun getTitleText(): String {
        return title.toString()
    }

    /**
     * 设置标题内容
     */
    fun setTitleText(titleStr: String) {
        getToolbar().title = titleStr
    }

    /**
     * 获取标题颜色，可重新自定义设置标题文本颜色
     */
    open fun getTitleTextColor(): Int {
        return RxKit.getTitleTextColor()
    }

    fun setNavigationIcon(iconId: Int) {
        toolbar.setNavigationIcon(iconId)
    }

    fun getNavigationIcon() = getToolbar().navigationIcon

    fun showLoadingDialog(show: Boolean) {
        if (loadingDialog == null) {
            loadingDialog = Dialog(this)
        }
        loadingDialog!!.window.setBackgroundDrawableResource(android.R.color.transparent)
        loadingDialog!!.window.setDimAmount(0f)
        loadingDialog!!.setCanceledOnTouchOutside(false)
        val loadingView = LoadingView(this)
        loadingView.setColor(RxKit.getLoadingDialogColor())
        loadingView.setSize(DensityUtils.dp2px(RxKit.getLoadingDialogSize()))
        loadingDialog!!.setContentView(loadingView)
        if (show) {
            if (!loadingDialog!!.isShowing) {
                loadingDialog!!.show()
            }
        } else {
            if (loadingDialog!!.isShowing) {
                loadingDialog!!.dismiss()
            }
        }
    }

    override fun onPrepareOptionsMenu(menu: Menu?): Boolean {
        return super.onPrepareOptionsMenu(menu)
    }

    override fun onResume() {
        super.onResume()
        for (i in 0 until getMenuCount()) {
            val menuItem = getMenuItem(i)
            menuItem.icon?.setColorFilter(getTitleTextColor(), PorterDuff.Mode.SRC_ATOP)
            val titleStr = menuItem.title
            if (!TextUtils.isEmpty(titleStr)) {
                menuItem.title = addColor(titleStr, getTitleTextColor())
            }
        }
        getToolbar().navigationIcon?.setColorFilter(getTitleTextColor(), PorterDuff.Mode.SRC_ATOP)
    }

    fun getMenuItem(position: Int): MenuItem {
        return toolbar.menu.getItem(position)
    }

    fun getMenuCount() = toolbar.menu.size()

    private fun addColor(text: CharSequence, color: Int): SpannableStringBuilder {
        val builder = SpannableStringBuilder(text)
        if (color != 0) {
            builder.setSpan(
                ForegroundColorSpan(color), 0, text.length,
                Spannable.SPAN_EXCLUSIVE_EXCLUSIVE
            )
        }
        return builder
    }

    /**
     * 设置状态栏黑色字体和图标
     */
    fun setStatusBarLightMode() {
        if (ImmersionBar.isSupportStatusBarDarkFont()) {
            ImmersionBar.with(this)
                .statusBarDarkFont(true)
                .supportActionBar(true)
                .fitsSystemWindows(true)
                .statusBarColorInt(RxKit.getTitleBackground())
                .init()
        }
    }

    /**
     * 设置状态栏白色字体和图标
     */
    fun setStatusBarDarkMode() {
        ImmersionBar.with(this)
            .statusBarDarkFont(false)
            .fitsSystemWindows(true)
            .supportActionBar(true)
            .statusBarColorInt(RxKit.getTitleBackground())
            .init()
    }

    fun setStatusBarMode(lightMode: Boolean) {
        if (lightMode) {
            setStatusBarLightMode()
        } else {
            setStatusBarDarkMode()
        }
    }

    /**
     * 显示加载布局
     */
    fun showLoading(show: Boolean) {
        if (show) {
            if (loadingView == null) {
                loadingView = LayoutInflater.from(this).inflate(RxKit.getLoadingLayout(), null)
                getRootViewGroup().addView(loadingView)
            }
        } else {
            getRootViewGroup().removeView(loadingView)
            loadingView = null
        }
    }

    /**
     * 显示空布局
     */
    fun showEmpty(show: Boolean) {
        if (show) {
            if (emptyView == null) {
                emptyView = LayoutInflater.from(this).inflate(RxKit.getEmptyLayout(), null)
                getRootViewGroup().addView(emptyView)
            }
        } else {
            getRootViewGroup().removeView(emptyView)
            emptyView = null
        }
    }

    private fun getRootViewGroup(): ViewGroup {
        return getRootView() as ViewGroup
    }

    private fun getRootView(): View {
        return window.decorView.findViewById(android.R.id.content)
    }

    protected open fun hideInputMethod() {
        val imm = this.getSystemService(
            Activity.INPUT_METHOD_SERVICE
        ) as InputMethodManager
        val view = currentFocus
        if (view != null) {
            imm.hideSoftInputFromWindow(view.windowToken, 0)
        }
    }

    override fun onOptionsItemSelected(item: MenuItem?): Boolean {
        val itemId = item!!.itemId
        if (itemId == android.R.id.home) {
            hideInputMethod()
            setNavigationIconClick()
        }
        return super.onOptionsItemSelected(item)
    }

    override fun onMenuOpened(featureId: Int, menu: Menu?): Boolean {
        if (menu != null) {
            if (menu.javaClass.simpleName == "MenuBuilder") {
                try {
                    val method = menu.javaClass
                        .getDeclaredMethod("setOptionalIconsVisible", java.lang.Boolean.TYPE)
                    method.isAccessible = true
                    method.invoke(menu, true)
                } catch (e: Exception) {
                    e.printStackTrace()
                }

            }
        }
        return super.onMenuOpened(featureId, menu)
    }
}