package rx.library.kit.http.builder;

import android.text.TextUtils;

import java.io.File;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

import okhttp3.CacheControl;
import okhttp3.Headers;
import okhttp3.Headers.Builder;
import okhttp3.RequestBody;
import rx.library.kit.http.ApiInitialize;
import rx.library.kit.http.ApiProgressListener;
import rx.library.kit.http.other.QXHttpBuildUtil;

public class BaseHttpBuilder<T> extends LinkedHashMap<String, Object> {
    /**
     * 接口地址
     */
    private String mUrl;
    private String simpleUrl;
    private Object mTag;
    private CacheControl mCacheControl;

    private Map<String, File> fileMap;

    private String mJsonParams;
    /**
     * 请求头构造器
     */
    private Builder mHBuilder;

    /**
     * 上传文件进度监听
     */
    private ApiProgressListener progressListener;

    protected final String SocketIsClosed = "Socket is closed";
    protected final String SocketClosed = "Socket closed";
    protected final String Canceled = "Canceled";
    protected final String Timeout = "timeout";

    public BaseHttpBuilder(String url) {
        simpleUrl = url;
    }

    public Headers getHeaders() {
        if (mHBuilder == null) {
            return null;
        } else {
            return mHBuilder.build();
        }
    }

    public String getHeader(String key) {
        return getHeadersBuilder().get(key);
    }


    public Builder getHeadersBuilder() {
        if (mHBuilder == null) {
            mHBuilder = new Builder();
        }
        return mHBuilder;
    }

    public T setHeadersBuilder(Builder builder) {
        mHBuilder = builder;
        return (T) this;
    }


    public T addHeader(String key, String value) {
        getHeadersBuilder().add(key, value);
        return (T) this;
    }

    public T addHeader(Map<String, String> headerMap) {
        if (headerMap != null) {
            for (Entry<String, String> entry : headerMap.entrySet()) {
                String key = entry.getKey();
                String value = entry.getValue();
                getHeadersBuilder().add(key, value);
            }
        }
        return (T) this;
    }

    public T addHeader(String line) {
        getHeadersBuilder().add(line);
        return (T) this;
    }

    public T setHeader(String key, String value) {
        getHeadersBuilder().set(key, value);
        return (T) this;
    }

    public T removeAllHeader(String key) {
        getHeadersBuilder().removeAll(key);
        return (T) this;
    }

    public T add(String key, Object value) {
        if (value == null) {
            value = "";
        }
        super.put(key, value);
        return (T) this;
    }

    public T setProgressListener(ApiProgressListener progressListener) {
        this.progressListener = progressListener;
        return (T) this;
    }

    public T add(String key, File value) {
        if (fileMap == null) {
            fileMap = new LinkedHashMap<>();
        }
        fileMap.put(key, value);
        return (T) this;
    }

    public T add(Map<String, Object> map) {
        if (map != null) {
            super.putAll(map);
        }
        return (T) this;
    }

    public T setJsonParams(String jsonParams) {
        mJsonParams = jsonParams;
        return (T) this;
    }

    public String getJsonParams() {
        return mJsonParams;
    }

    public Map<String, Object> getParams() {
        return this;
    }

    public String getGetParams() {
        String getParams = "";
        for (Entry<String, Object> entry : entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue().toString();
            if (TextUtils.isEmpty(getParams)) {
                getParams = key.concat("=".concat(value));
            } else {
                getParams = getParams.concat("&".concat(key.concat("=".concat(value))));
            }
        }
        return getParams;
    }

    public Map<String, File> getFileMap() {
        return fileMap == null ? (new HashMap<String, File>()) : fileMap;
    }

    public T setUrl(String url) {
        this.mUrl = url;
        return (T) this;
    }

    public String getUrl() {
        if (TextUtils.isEmpty(mUrl)) {
            if (TextUtils.isEmpty(ApiInitialize.getBaseUrl())) {
                throw new IllegalArgumentException("baseUrl cannot be empty");
            } else {
                return ApiInitialize.getBaseUrl() + simpleUrl;
            }
        } else {
            return mUrl;
        }
    }

    public String getSimpleUrl() {
        if (TextUtils.isEmpty(mUrl)) {
            return simpleUrl;
        } else {
            return mUrl;
        }
    }

    public T addTag(Object tag) {
        mTag = tag;
        return (T) this;
    }

    public Object getTag() {
        return mTag;
    }

    public CacheControl getCacheControl() {
        return mCacheControl;
    }

    public T cacheControl(CacheControl cacheControl) {
        mCacheControl = cacheControl;
        return (T) this;
    }

    public RequestBody getFileRequestBody() {
        RequestBody requestBody;
        if (fileMap == null) {
            requestBody = QXHttpBuildUtil.buildFormRequestBody(this);
        } else {
            requestBody = QXHttpBuildUtil.buildFormRequestBody(this, fileMap, progressListener);
        }
        return requestBody;
    }

    public RequestBody getRequestBody() {
        return QXHttpBuildUtil.buildFormRequestBody(this);
    }

}