package rx.library.kit.dialog;

import android.app.Dialog;
import android.content.Context;
import android.graphics.Color;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.GradientDrawable;
import android.os.Build;
import android.text.TextUtils;
import android.view.LayoutInflater;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.widget.TextView;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import rx.library.kit.R;
import rx.library.kit.dialog.interfaces.OnDatePickerListener;
import rx.library.kit.util.RxDeviceInfo;

public class RxDatePickerDialog {
    public static Builder newBuilder(Context context) {
        return new Builder(context);
    }

    public static class Builder {
        private Context context;
        private String titleStr;
        private String year;
        private String month;
        private String day;
        private String hour = "00";
        private String minute = "00";
        private String second = "00";
        private RxDatePickerView pickerYear;
        private RxDatePickerView pickerMonth;
        private RxDatePickerView pickerDay;
        private RxDatePickerView pickerHour;
        private RxDatePickerView pickerMinute;
        private RxDatePickerView pickerSecond;

        private boolean showDate = true;
        private boolean showTime = false;

        public Builder(Context context) {
            this.context = context;
        }

        /**
         * 设置标题
         *
         * @param title
         */
        public void setTitle(String title) {
            this.titleStr = title;
        }

        /**
         * 初始化日期
         *
         * @param dateStr
         * @return
         */
        public Builder setDate(String dateStr) {
            try {
                String[] split = dateStr.split(" ");
                String[] date = split[0].split("-");
                String[] time = split[1].split(":");
                year = date[0];
                month = date[1];
                day = date[2];
                hour = time[0];
                minute = time[1];
                second = time[2];
            } catch (Exception e) {

            }
            return this;
        }

        /**
         * 设置是否显示年月日
         *
         * @param showDate true：显示； false：隐藏
         * @return
         */
        public Builder showDate(boolean showDate) {
            this.showDate = showDate;
            return this;
        }

        /**
         * 设置是否显示时分秒
         *
         * @param showTime true：显示； false：隐藏
         * @return
         */
        public Builder showTime(boolean showTime) {
            this.showTime = showTime;
            return this;
        }

        public Dialog build(final OnDatePickerListener listener) {

            checkDate();

            final Dialog dialog = new Dialog(context);
            View view = LayoutInflater.from(context).inflate(R.layout.kit_date_picker_dialog, null);
            if (!TextUtils.isEmpty(titleStr)) {
                TextView titleView = view.findViewById(R.id.kit_date_picker_dialog_title);
                titleView.setText(titleStr);
            }
            pickerYear = view.findViewById(R.id.kit_date_picker_dialog_year);
            pickerYear.setIsLoop(false);
            pickerYear.setData(getDataList(1970, 2100));
            pickerYear.setSelected(year);

            pickerMonth = view.findViewById(R.id.kit_date_picker_dialog_month);
            pickerMonth.setIsLoop(false);
            pickerMonth.setData(getDataList(1, 13));
            pickerMonth.setSelected(month);

            pickerDay = view.findViewById(R.id.kit_date_picker_dialog_day);
            pickerDay.setIsLoop(false);
            pickerDay.setData(getDataList(1, 32));
            pickerDay.setSelected(day);

            pickerHour = view.findViewById(R.id.kit_date_picker_dialog_hour);
            pickerHour.setIsLoop(false);
            pickerHour.setData(getDataList(0, 24));
            pickerHour.setSelected(hour);

            pickerMinute = view.findViewById(R.id.kit_date_picker_dialog_minute);
            pickerMinute.setIsLoop(false);
            pickerMinute.setData(getDataList(0, 60));
            pickerMinute.setSelected(minute);

            pickerSecond = view.findViewById(R.id.kit_date_picker_dialog_second);
            pickerSecond.setIsLoop(false);
            pickerSecond.setData(getDataList(0, 60));
            pickerSecond.setSelected(second);

            pickerYear.setOnSelectListener(yearListener);
            pickerMonth.setOnSelectListener(monthListener);
            pickerDay.setOnSelectListener(dayListener);
            pickerHour.setOnSelectListener(hourListener);
            pickerMinute.setOnSelectListener(minuteListener);
            pickerSecond.setOnSelectListener(secondListener);
            if (showDate) {
                view.findViewById(R.id.kit_date_picker_dialog_date).setVisibility(View.VISIBLE);
            } else {
                view.findViewById(R.id.kit_date_picker_dialog_date).setVisibility(View.GONE);
            }
            if (showTime) {
                view.findViewById(R.id.kit_date_picker_dialog_time).setVisibility(View.VISIBLE);
            } else {
                view.findViewById(R.id.kit_date_picker_dialog_time).setVisibility(View.GONE);
            }
            view.findViewById(R.id.kit_date_picker_dialog_cancel).setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    dialog.cancel();
                }
            });
            view.findViewById(R.id.kit_date_picker_dialog_ok).setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    dialog.cancel();
                    if (listener != null) {
                        if (showTime) {
                            String selectedStr = year + "-" + month + "-" + day + " " + hour + ":" + minute + ":" + second;
                            listener.onSelectedListener(selectedStr);
                        } else {
                            String selectedStr = year + "-" + month + "-" + day;
                            listener.onSelectedListener(selectedStr);
                        }
                    }
                }
            });
            GradientDrawable drawable = new GradientDrawable();
            if (showTime) {
                drawable.setCornerRadius(0);
            } else {
                drawable.setCornerRadius(40);
            }
            drawable.setColor(Color.WHITE);
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                view.setBackground(drawable);
            } else {
                view.setBackgroundDrawable(drawable);
            }
            dialog.setContentView(view);

            Window window = dialog.getWindow();
            assert window != null;
            WindowManager.LayoutParams lp = window.getAttributes();
            window.setBackgroundDrawable(new ColorDrawable(Color.TRANSPARENT));
            if (showTime) {
                lp.width = RxDeviceInfo.getScreenWidth();
            } else {
                lp.width = RxDeviceInfo.getScreenWidth() * 7 / 10;
            }
            dialog.getWindow().setAttributes(lp);
            dialog.show();
            return dialog;
        }

        private void checkDate() {
            if (TextUtils.isEmpty(year) || TextUtils.isEmpty(month) || TextUtils.isEmpty(day)) {
                Calendar calendar = Calendar.getInstance();
                year = String.valueOf(calendar.get(Calendar.YEAR));
                int m = calendar.get(Calendar.MONTH) + 1;
                month = m < 10 ? "0" + m : String.valueOf(m);
                int d = calendar.get(Calendar.DATE);
                day = d < 10 ? "0" + d : String.valueOf(d);
                int h = calendar.get(Calendar.HOUR);
                hour = h < 10 ? "0" + h : String.valueOf(h);
                int min = calendar.get(Calendar.MINUTE);
                minute = min < 10 ? "0" + min : String.valueOf(min);
                int s = calendar.get(Calendar.SECOND);
                second = s < 10 ? "0" + s : String.valueOf(s);
            }
        }

        private RxDatePickerView.onSelectListener yearListener = new RxDatePickerView.onSelectListener() {
            @Override
            public void onSelect(String text) {
                year = text;
            }
        };
        private RxDatePickerView.onSelectListener monthListener = new RxDatePickerView.onSelectListener() {
            @Override
            public void onSelect(String text) {
                month = text;
                int monthLastDay = getMonthLastDay(Integer.valueOf(year), Integer.valueOf(month));
                pickerDay.setData(getDataList(1, monthLastDay + 1));
                pickerDay.setSelected(0);
            }
        };
        private RxDatePickerView.onSelectListener dayListener = new RxDatePickerView.onSelectListener() {
            @Override
            public void onSelect(String text) {
                day = text;
            }
        };
        private RxDatePickerView.onSelectListener hourListener = new RxDatePickerView.onSelectListener() {
            @Override
            public void onSelect(String text) {
                hour = text;
            }
        };
        private RxDatePickerView.onSelectListener minuteListener = new RxDatePickerView.onSelectListener() {
            @Override
            public void onSelect(String text) {
                minute = text;
            }
        };
        private RxDatePickerView.onSelectListener secondListener = new RxDatePickerView.onSelectListener() {
            @Override
            public void onSelect(String text) {
                second = text;
            }
        };

        private List<String> getDataList(int start, int end) {
            List<String> list = new ArrayList<>();
            for (int i = start; i < end; i++) {
                if (i < 10) {
                    list.add("0" + i);
                } else {
                    list.add(String.valueOf(i));
                }
            }
            return list;
        }

    }

    private static int getMonthLastDay(int year, int month) {
        Calendar a = Calendar.getInstance();
        a.set(Calendar.YEAR, year);
        a.set(Calendar.MONTH, month - 1);
        a.set(Calendar.DATE, 1);
        a.roll(Calendar.DATE, -1);
        int maxDate = a.get(Calendar.DATE);
        return maxDate;
    }

}
