package rx.library.kit.dialog;

import android.app.Dialog;
import android.content.Context;
import android.content.res.ColorStateList;
import android.graphics.Color;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.GradientDrawable;
import android.os.Build;
import android.text.TextUtils;
import android.view.*;
import android.widget.LinearLayout;
import android.widget.TextView;
import androidx.appcompat.content.res.AppCompatResources;
import rx.library.kit.R;
import rx.library.kit.dialog.interfaces.OnDialogClickListener;
import rx.library.kit.util.RxDeviceInfo;

import java.util.ArrayList;
import java.util.List;

public class RxDialog {

    public static Builder newBuilder(Context context) {
        return new Builder(context);
    }

    public static class Builder {
        private Context context;
        private String mTitle;
        private String mMessage;
        private int mGravity = Gravity.CENTER;
        private boolean allWidth = false;
        private List<ButtonView> list;

        public Builder(Context context) {
            this.context = context;
        }

        public Builder setTitle(String title) {
            this.mTitle = title;
            return this;
        }

        public Builder setMessage(String message) {
            this.mMessage = message;
            return this;
        }

        public Builder addButton(String butString) {
            addButton(butString, null);
            return this;
        }

        public Builder addButton(String butString, int textColor) {
            addButton(butString, textColor, null);
            return this;
        }

        public Builder addButton(String butString, OnDialogClickListener listener) {
            addButton(butString, R.color.kit_dialog_textview, listener);
            return this;
        }


        public Builder addButton(String butString, int textColor, final OnDialogClickListener listener) {
            if (list == null) {
                list = new ArrayList<>();
            }
            TextView textView = new TextView(context);
            textView.setText(butString);
            textView.setClickable(true);
            ColorStateList colorStateList = AppCompatResources.getColorStateList(context, textColor);
            textView.setTextColor(colorStateList);
            textView.setGravity(Gravity.CENTER);
            ButtonView buttonView = new ButtonView();
            buttonView.setTextView(textView);
            buttonView.setListener(listener);
            list.add(buttonView);
            return this;
        }

        public Builder asAllWidth() {
            this.allWidth = true;
            return this;
        }

        public Builder setGravity(int gravity) {
            this.mGravity = gravity;
            return this;
        }

        public Dialog build() {
            final Dialog dialog = new Dialog(context);
            View view = LayoutInflater.from(context).inflate(R.layout.kit_dialog, null);
            if (!TextUtils.isEmpty(mTitle)) {
                TextView titleView = view.findViewById(R.id.kit_ui_dialog_title);
                titleView.setText(mTitle);
            }
            TextView messageView = view.findViewById(R.id.kit_ui_dialog_message);
            if (!TextUtils.isEmpty(mMessage)) {
                messageView.setText(mMessage);
            } else {
                messageView.setVisibility(View.GONE);
            }
            final LinearLayout linearLayout = view.findViewById(R.id.kit_ui_dialog_layout);
            for (int i = 0; i < list.size(); i++) {
                if (i != 0) {
                    View view1 = new View(context);
                    view1.setBackgroundColor(Color.parseColor("#f3f3f3"));
                    linearLayout.addView(view1);
                    LinearLayout.LayoutParams llp = (LinearLayout.LayoutParams) view1.getLayoutParams();
                    llp.width = 2;
                    view1.setLayoutParams(llp);
                }
                final View v = list.get(i).getTextView();
                LinearLayout.LayoutParams lp = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);
                lp.weight = 1;
                v.setLayoutParams(lp);
                final OnDialogClickListener listener = list.get(i).getListener();
                v.setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        dialog.cancel();
                        if (listener != null) {
                            listener.onClick();
                        }
                    }
                });
                linearLayout.addView(v);
            }
            if (allWidth) {
                view.setBackgroundColor(Color.WHITE);
            } else {
                GradientDrawable drawable = new GradientDrawable();
                drawable.setCornerRadius(40);
                drawable.setColor(Color.WHITE);
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                    view.setBackground(drawable);
                } else {
                    view.setBackgroundDrawable(drawable);
                }
            }
            dialog.setContentView(view);
            Window window = dialog.getWindow();
            assert window != null;

            WindowManager.LayoutParams lp = window.getAttributes();
            window.setBackgroundDrawable(new ColorDrawable(Color.TRANSPARENT));
            if (allWidth) {
                lp.width = RxDeviceInfo.getScreenWidth();
            } else {
                lp.width = RxDeviceInfo.getScreenWidth() * 7 / 10;
            }
            dialog.getWindow().setAttributes(lp);
            window.setGravity(mGravity);
            dialog.show();
            return dialog;
        }
    }

    private static class ButtonView {
        private TextView textView;
        private OnDialogClickListener listener;

        public TextView getTextView() {
            return textView;
        }

        public void setTextView(TextView textView) {
            this.textView = textView;
        }

        public OnDialogClickListener getListener() {
            return listener;
        }

        public void setListener(OnDialogClickListener listener) {
            this.listener = listener;
        }
    }
}
