package android.library.util.http;

import android.library.util.LogUtil;
import androidx.annotation.NonNull;
import io.reactivex.Observable;
import io.reactivex.ObservableEmitter;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.Observer;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.disposables.Disposable;
import io.reactivex.schedulers.Schedulers;
import okhttp3.*;
import okhttp3.Headers.Builder;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.LinkedHashMap;
import java.util.Map;

public abstract class HttpBuilder extends LinkedHashMap<String, Object> {
    /**
     * 接口地址
     */
    private String mUrl;
    private Object mTag;
    private CacheControl mCacheControl;

    private Map<String, File> fileMap;

    private String mJsonParams;
    /**
     * 请求头构造器
     */
    private Builder mHBuilder;

    public HttpBuilder(String url) {
        mUrl = url;
    }

    public Headers getHeaders() {
        if (mHBuilder == null) {
            return null;
        } else {
            return mHBuilder.build();
        }
    }


    public String getHeader(String key) {
        return getHeadersBuilder().get(key);
    }


    public Builder getHeadersBuilder() {
        if (mHBuilder == null) {
            mHBuilder = new Builder();
        }
        return mHBuilder;
    }


    public HttpBuilder setHeadersBuilder(Builder builder) {
        mHBuilder = builder;
        return this;
    }


    public HttpBuilder addHeader(String key, String value) {
        getHeadersBuilder().add(key, value);
        return this;
    }

    public HttpBuilder addHeader(Map<String, String> headerMap) {
        if (headerMap != null) {
            for (Entry<String, String> entry : headerMap.entrySet()) {
                String key = entry.getKey();
                String value = entry.getValue();
                getHeadersBuilder().add(key, value);
            }
        }
        return this;
    }


    public HttpBuilder addHeader(String line) {
        getHeadersBuilder().add(line);
        return this;
    }


    public HttpBuilder setHeader(String key, String value) {
        getHeadersBuilder().set(key, value);
        return this;
    }


    public HttpBuilder removeAllHeader(String key) {
        getHeadersBuilder().removeAll(key);
        return this;
    }


    public HttpBuilder setUrl(@NonNull String url) {
        mUrl = url;
        return null;
    }


    public HttpBuilder add(String key, Object value) {
        if (value == null) {
            value = "";
        }
        super.put(key, value);
        return this;
    }


    public HttpBuilder add(String key, File value) {
        if (fileMap == null) {
            fileMap = new LinkedHashMap<>();
        }
        fileMap.put(key, value);
        return this;
    }


    public HttpBuilder add(Map<String, Object> map) {
        if (map != null) {
            super.putAll(map);
        }
        return this;
    }


    public HttpBuilder setJsonParams(String jsonParams) {
        mJsonParams = jsonParams;
        return this;
    }

    public String getJsonParams() {
        return mJsonParams;
    }

    public Map<String, Object> getParams() {
        return this;
    }


    public String getUrl() {
        return mUrl;
    }


    public String getSimpleUrl() {
        return mUrl;
    }


    public HttpBuilder addTag(Object tag) {
        mTag = tag;
        return this;
    }


    public Object getTag() {
        return mTag;
    }


    public CacheControl getCacheControl() {
        return mCacheControl;
    }

    public HttpBuilder cacheControl(CacheControl cacheControl) {
        mCacheControl = cacheControl;
        return this;
    }

    public RequestBody getRequestBody() {
        RequestBody requestBody;
        if (fileMap == null) {
            requestBody = HttpBuildUtil.buildFormRequestBody(this);
        } else {
            requestBody = HttpBuildUtil.buildFormRequestBody(this, fileMap);
        }
        return requestBody;
    }

    public abstract void builder(ApiCallBack apiCallBack);

    public void request(final Request request, final ApiCallBack apiCallBack) {
        Observable.create(new ObservableOnSubscribe<HttpResponse>() {
            @Override
            public void subscribe(final ObservableEmitter<HttpResponse> emitter) throws Exception {
                HttpSender.execute(request, new Callback() {
                    @Override
                    public void onFailure(Call call, IOException e) {
                        Request failRequest = call.request();
                        ErrorThrowable errorThrowable = new ErrorThrowable();
                        try {
                            errorThrowable.setUrl(URLDecoder.decode(failRequest.url().toString(), "utf-8"));
                        } catch (UnsupportedEncodingException e1) {
                            e1.printStackTrace();
                            errorThrowable.setUrl(failRequest.url().toString());
                        }
                        errorThrowable.setMethod(failRequest.method());
                        errorThrowable.setMessage(e.getMessage());
                        emitter.onError(errorThrowable);
                        emitter.onComplete();
                    }

                    @Override
                    public void onResponse(Call call, Response response) throws IOException {
                        boolean successful = response.isSuccessful();
                        String mUrl = URLDecoder.decode(request.url().toString(), "utf-8");
                        int code = response.code();
                        String method = request.method();
                        HttpResponse httpResponse;
                        if (successful && code == 200) {
                            httpResponse = new HttpResponse(mUrl, response.body().string(), true, method, code);
                        } else {
                            httpResponse = new HttpResponse(mUrl, response.body().string(), false, method, code);
                        }
                        emitter.onNext(httpResponse);
                        emitter.onComplete();
                    }
                });
            }
        })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribeOn(Schedulers.io())
                .subscribe(new Observer<HttpResponse>() {
                    @Override
                    public void onSubscribe(Disposable d) {
                        try {
                            String mUrl = URLDecoder.decode(request.url().toString(), "utf-8");
                            apiCallBack.before(mUrl, request.method());
                        } catch (UnsupportedEncodingException e) {
                            e.printStackTrace();
                            apiCallBack.before(request.url().toString(), request.method());
                        }

                    }

                    @Override
                    public void onNext(HttpResponse response) {
                        if (response.isSuccess()) {
                            apiCallBack.success(response.getUrl(), response.getData(), response.getMethod());
                        } else {
                            apiCallBack.error(response.getUrl(), response.getData(), response.getMethod());
                        }
                        HttpLogUtil.log(HttpBuilder.this, response);
                    }

                    @Override
                    public void onError(Throwable e) {
                        ErrorThrowable throwable = (ErrorThrowable) e;
                        String message = throwable.getMessage();
                        String errorUrl = throwable.getUrl();
                        String errorMethod = throwable.getMethod();
                        apiCallBack.error(errorUrl, message, errorMethod);
                    }

                    @Override
                    public void onComplete() {
                        String url = request.url().toString();
                        try {
                            url = URLDecoder.decode(url, "utf-8");
                        } catch (UnsupportedEncodingException e) {
                            e.printStackTrace();
                        }
                        apiCallBack.onComplete(url, request.method());
                    }
                });


    }


}