package android.library.util.http.builder;

import android.library.util.http.*;
import android.library.util.http.other.QXErrorThrowable;
import android.library.util.http.other.QXHttpLogUtil;
import android.library.util.http.other.QXHttpResponse;
import android.library.util.http.other.QXHttpSender;
import io.reactivex.Observable;
import io.reactivex.ObservableEmitter;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.Observer;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.disposables.Disposable;
import io.reactivex.schedulers.Schedulers;
import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Request;
import okhttp3.Response;

import java.io.IOException;

public abstract class QXHttpBuilder extends QXBaseHttpBuilder<QXHttpBuilder> {
    public QXHttpBuilder(String url) {
        super(url);
    }

    public abstract void builder(ApiCallBack apiCallBack);

    public void request(final Request request, final ApiCallBack apiCallBack) {
        Observable.create(new ObservableOnSubscribe<QXHttpResponse>() {
            @Override
            public void subscribe(final ObservableEmitter<QXHttpResponse> emitter) throws Exception {
                QXHttpLogUtil.logHeader(getSimpleUrl(), getHeaders());
                QXHttpLogUtil.logParams(getSimpleUrl(), getParams());

                QXHttpSender.execute(request, new Callback() {
                    @Override
                    public void onFailure(Call call, IOException e) {
                        Request failRequest = call.request();
                        QXErrorThrowable errorThrowable = new QXErrorThrowable();
                        errorThrowable.setUrl(getSimpleUrl());
                        errorThrowable.setMethod(failRequest.method());
                        errorThrowable.setMessage(e.getMessage());
                        emitter.onError(errorThrowable);
                        emitter.onComplete();
                    }

                    @Override
                    public void onResponse(Call call, Response response) throws IOException {
                        boolean successful = response.isSuccessful();
                        int code = response.code();
                        String method = response.request().method();
                        QXHttpResponse httpResponse;
                        if (successful && code == 200) {
                            httpResponse = new QXHttpResponse(getSimpleUrl(), response.body().string(), true, method, code);
                        } else {
                            httpResponse = new QXHttpResponse(getSimpleUrl(), response.body().string(), false, method, code);
                        }
                        emitter.onNext(httpResponse);
                        emitter.onComplete();
                    }
                });
            }
        })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribeOn(Schedulers.io())
                .subscribe(new Observer<QXHttpResponse>() {
                    @Override
                    public void onSubscribe(Disposable d) {
                        apiCallBack.before(getSimpleUrl(), request.method());
                    }

                    @Override
                    public void onNext(QXHttpResponse response) {
                        if (response.isSuccess()) {
                            QXHttpLogUtil.log("成功", getSimpleUrl(), response.getData());
                            apiCallBack.success(response.getUrl(), response.getData(), response.getMethod());
                        } else {
                            QXHttpLogUtil.log("失败", getSimpleUrl(), "code:" + response.getCode());
                            apiCallBack.error(response.getUrl(), response.getData(), response.getMethod());
                        }
                    }

                    @Override
                    public void onError(Throwable e) {
                        QXErrorThrowable throwable = (QXErrorThrowable) e;
                        String message = throwable.getMessage();
                        String errorUrl = throwable.getUrl();
                        String errorMethod = throwable.getMethod();
                        if (SocketClosed.equals(message) || SocketIsClosed.equals(message) || Canceled.equals(message)) {
                            apiCallBack.closed(errorUrl, errorMethod);
                            QXHttpLogUtil.log("取消", getSimpleUrl(), "请求被取消");
                            apiCallBack.complete(getSimpleUrl(), errorMethod);
                            QXHttpLogUtil.log("结束", getSimpleUrl(), "接口请求结束");
                        } else {
                            apiCallBack.error(errorUrl, message, errorMethod);
                            QXHttpLogUtil.log("错误", getSimpleUrl(), e.getMessage());
                        }
                    }

                    @Override
                    public void onComplete() {
                        QXHttpLogUtil.log("完成", getSimpleUrl(), "接口请求完成");
                        apiCallBack.complete(getSimpleUrl(), request.method());
                    }
                });


    }


}
