package android.library.util.http.builder;

import android.library.util.http.ApiDownloadCallBack;
import android.library.util.http.other.QXErrorThrowable;
import android.library.util.http.other.QXHttpLogUtil;
import android.library.util.http.other.QXHttpSender;
import android.text.TextUtils;
import io.reactivex.Observable;
import io.reactivex.ObservableEmitter;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.Observer;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.disposables.Disposable;
import io.reactivex.schedulers.Schedulers;
import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Request;
import okhttp3.Response;

import java.io.*;

public abstract class QXDownloadHttpBuilder extends QXBaseHttpBuilder<QXDownloadBuilderHttp> {

    private long startsPoint = 0;

    public QXDownloadHttpBuilder(String url) {
        super(url);
    }

    public abstract void builder(String path, ApiDownloadCallBack apiDownloadCallBack);

    public void downloading(final String path, final Request request, final ApiDownloadCallBack apiDownloadCallBack) {
        Observable.create(new ObservableOnSubscribe<Result>() {
            @Override
            public void subscribe(final ObservableEmitter<Result> emitter) throws Exception {
                QXHttpLogUtil.logHeader(getSimpleUrl(), getHeaders());
                QXHttpLogUtil.logParams(getSimpleUrl(), getParams());

                QXHttpSender.executeDownload(request, new Callback() {
                    @Override
                    public void onFailure(Call call, IOException e) {
                        Request failRequest = call.request();
                        QXErrorThrowable errorThrowable = new QXErrorThrowable();
                        errorThrowable.setUrl(getSimpleUrl());
                        errorThrowable.setMethod(failRequest.method());
                        errorThrowable.setMessage(e.getMessage());
                        emitter.onError(errorThrowable);
                        emitter.onComplete();
                    }

                    @Override
                    public void onResponse(Call call, Response response) throws IOException {
                        boolean successful = response.isSuccessful();
                        if (successful) {
                            String fileName = getHeaderFileName(response);
                            long length = response.body().contentLength();
                            if (length == 0) {
                                emitter.onNext(new Result("下载失败", false));
                                emitter.onComplete();
                                return;
                            }
                            /**
                             * 保存文件到本地
                             */
                            InputStream is = null;
                            RandomAccessFile randomAccessFile = null;
                            BufferedInputStream bis = null;

                            byte[] buff = new byte[2048];
                            int len = 0;
                            try {
                                is = response.body().byteStream();
                                bis = new BufferedInputStream(is);
                                File file = getFile(path, fileName);
                                // 随机访问文件，可以指定断点续传的起始位置
                                randomAccessFile = new RandomAccessFile(file, "rwd");
//                                randomAccessFile.seek (startsPoint);
                                while ((len = bis.read(buff)) != -1) {
                                    randomAccessFile.write(buff, 0, len);
                                }

                                // 下载完成
                                emitter.onNext(new Result(true, file.getPath()));
                                emitter.onComplete();
                            } catch (Exception e) {
                                e.printStackTrace();
                                QXErrorThrowable errorThrowable = new QXErrorThrowable();
                                errorThrowable.setUrl(getSimpleUrl());
                                errorThrowable.setMessage(e.getMessage());
                                emitter.onError(errorThrowable);
                                emitter.onComplete();

                            } finally {
                                try {
                                    if (is != null) {
                                        is.close();
                                    }
                                    if (bis != null) {
                                        bis.close();
                                    }
                                    if (randomAccessFile != null) {
                                        randomAccessFile.close();
                                    }
                                } catch (Exception e) {
                                    e.printStackTrace();
                                }
                            }
                        } else {
                            emitter.onNext(new Result(response.body().string(), false));
                        }
                        emitter.onComplete();
                    }
                }, apiDownloadCallBack);
            }
        })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribeOn(Schedulers.io())
                .subscribe(new Observer<Result>() {
                    @Override
                    public void onSubscribe(Disposable d) {
                        apiDownloadCallBack.start(getSimpleUrl());
                    }

                    @Override
                    public void onNext(Result result) {
                        if (result.isSuccess()) {
                            QXHttpLogUtil.log("成功", getSimpleUrl(), "文件保存地址：" + result.getFilePath());
                            apiDownloadCallBack.success(getSimpleUrl(), result.getFilePath());
                        } else {
                            QXHttpLogUtil.log("失败", getSimpleUrl(), result.getErrorMsg());
                            apiDownloadCallBack.error(getSimpleUrl(), result.getErrorMsg());
                        }
                    }

                    @Override
                    public void onError(Throwable e) {
                        QXErrorThrowable throwable = (QXErrorThrowable) e;
                        String message = throwable.getMessage();
                        String errorUrl = throwable.getUrl();
                        String errorMethod = throwable.getMethod();
                        if (SocketClosed.equals(message) || SocketIsClosed.equals(message) || Canceled.equals(message)) {
                            apiDownloadCallBack.closed(errorUrl);
                            QXHttpLogUtil.log("取消", getSimpleUrl(), "下载被取消");
                            apiDownloadCallBack.complete(getSimpleUrl(), errorMethod);
                            QXHttpLogUtil.log("结束", getSimpleUrl(), "文件下载结束");
                        } else {
                            apiDownloadCallBack.error(errorUrl, message);
                            QXHttpLogUtil.log("错误", getSimpleUrl(), message);
                        }
                    }

                    @Override
                    public void onComplete() {
                        QXHttpLogUtil.log("完成", getSimpleUrl(), "文件下载完成");
                        apiDownloadCallBack.complete(getSimpleUrl(), request.method());
                    }
                });
    }

    private File getFile(String rootPath, String fileName) {
        File dir = new File(rootPath);
        if (!dir.exists()) {
            dir.mkdirs();
        }
        File file = new File(rootPath, fileName);
        file.delete();
        return file;
    }

    private static String getHeaderFileName(Response response) {
        String dispositionHeader = response.header("Content-Disposition");
        if (!TextUtils.isEmpty(dispositionHeader)) {
            dispositionHeader.replace("attachment;filename=", "");
            dispositionHeader.replace("filename*=utf-8", "");
            String[] strings = dispositionHeader.split(";");
            if (strings.length > 1) {
                dispositionHeader = strings[1].replace("filename=", "");
                dispositionHeader = dispositionHeader.replace("\"", "");
                return dispositionHeader;
            }
            return "";
        }
        return "";
    }

    private class Result {
        private String errorMsg;
        private boolean success;
        private String filePath;

        public Result(boolean success) {
            this.success = success;
        }

        public Result(String errorMsg, boolean success) {
            this.errorMsg = errorMsg;
            this.success = success;
        }

        public Result(boolean success, String filePath) {
            this.success = success;
            this.filePath = filePath;
        }

        public String getErrorMsg() {
            return errorMsg;
        }

        public void setErrorMsg(String errorMsg) {
            this.errorMsg = errorMsg;
        }

        public boolean isSuccess() {
            return success;
        }

        public void setSuccess(boolean success) {
            this.success = success;
        }

        public String getFilePath() {
            return filePath;
        }

        public void setFilePath(String filePath) {
            this.filePath = filePath;
        }
    }
}
