package android.library.util.http;

import android.text.TextUtils;
import io.reactivex.Observable;
import io.reactivex.ObservableEmitter;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.Observer;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.disposables.Disposable;
import io.reactivex.schedulers.Schedulers;
import okhttp3.*;
import okhttp3.Headers.Builder;

import java.io.File;
import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.Map;

public abstract class QXHttpBuilder extends LinkedHashMap<String, Object> {
    /**
     * 接口地址
     */
    private String mUrl;
    private String simpleUrl;
    private Object mTag;
    private CacheControl mCacheControl;

    private Map<String, File> fileMap;

    private String mJsonParams;
    /**
     * 请求头构造器
     */
    private Builder mHBuilder;

    private final String SocketIsClosed = "Socket is closed";
    private final String SocketClosed = "Socket closed";
    private final String Canceled = "Canceled";

    public QXHttpBuilder(String url) {
        simpleUrl = url;
    }

    public Headers getHeaders() {
        if (mHBuilder == null) {
            return null;
        } else {
            return mHBuilder.build();
        }
    }

    public String getHeader(String key) {
        return getHeadersBuilder().get(key);
    }


    public Builder getHeadersBuilder() {
        if (mHBuilder == null) {
            mHBuilder = new Builder();
        }
        return mHBuilder;
    }


    public QXHttpBuilder setHeadersBuilder(Builder builder) {
        mHBuilder = builder;
        return this;
    }


    public QXHttpBuilder addHeader(String key, String value) {
        getHeadersBuilder().add(key, value);
        return this;
    }

    public QXHttpBuilder addHeader(Map<String, String> headerMap) {
        if (headerMap != null) {
            for (Entry<String, String> entry : headerMap.entrySet()) {
                String key = entry.getKey();
                String value = entry.getValue();
                getHeadersBuilder().add(key, value);
            }
        }
        return this;
    }

    public QXHttpBuilder addHeader(String line) {
        getHeadersBuilder().add(line);
        return this;
    }

    public QXHttpBuilder setHeader(String key, String value) {
        getHeadersBuilder().set(key, value);
        return this;
    }

    public QXHttpBuilder removeAllHeader(String key) {
        getHeadersBuilder().removeAll(key);
        return this;
    }

    public QXHttpBuilder add(String key, Object value) {
        if (value == null) {
            value = "";
        }
        super.put(key, value);
        return this;
    }

    public QXHttpBuilder add(String key, File value) {
        if (fileMap == null) {
            fileMap = new LinkedHashMap<>();
        }
        fileMap.put(key, value);
        return this;
    }

    public QXHttpBuilder add(Map<String, Object> map) {
        if (map != null) {
            super.putAll(map);
        }
        return this;
    }

    public QXHttpBuilder setJsonParams(String jsonParams) {
        mJsonParams = jsonParams;
        return this;
    }

    public String getJsonParams() {
        return mJsonParams;
    }

    public Map<String, Object> getParams() {
        return this;
    }

    public String getGetParams() {
        String getParams = "";
        for (Entry<String, Object> entry : entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue().toString();
            if (TextUtils.isEmpty(getParams)) {
                if (value == null) {
                    getParams = key.concat("=");
                } else {
                    getParams = key.concat("=".concat(value));
                }
            } else {
                if (value == null) {
                    getParams = getParams.concat("&".concat(key.concat("=")));
                } else {
                    getParams = getParams.concat("&".concat(key.concat("=".concat(value))));
                }
            }
        }
        return getParams;
    }

    public QXHttpBuilder setUrl(String url) {
        this.mUrl = url;
        return this;
    }

    public String getUrl() {
        if (TextUtils.isEmpty(mUrl)) {
            if (TextUtils.isEmpty(ApiInitialize.getBaseUrl())) {
                throw new IllegalArgumentException("baseUrl cannot be empty");
            } else {
                return ApiInitialize.getBaseUrl() + simpleUrl;
            }
        } else {
            return mUrl;
        }
    }

    public String getSimpleUrl() {
        if (TextUtils.isEmpty(mUrl)) {
            return simpleUrl;
        } else {
            return mUrl;
        }
    }


    public QXHttpBuilder addTag(Object tag) {
        mTag = tag;
        return this;
    }


    public Object getTag() {
        return mTag;
    }


    public CacheControl getCacheControl() {
        return mCacheControl;
    }

    public QXHttpBuilder cacheControl(CacheControl cacheControl) {
        mCacheControl = cacheControl;
        return this;
    }

    public RequestBody getFileRequestBody() {
        RequestBody requestBody;
        if (fileMap == null) {
            requestBody = QXHttpBuildUtil.buildFormRequestBody(this);
        } else {
            requestBody = QXHttpBuildUtil.buildFormRequestBody(this, fileMap);
        }
        return requestBody;
    }

    public RequestBody getRequestBody() {
        return QXHttpBuildUtil.buildFormRequestBody(this);
    }

    public abstract void builder(ApiCallBack apiCallBack);

    public void request(final Request request, final ApiCallBack apiCallBack) {
        Observable.create(new ObservableOnSubscribe<QXHttpResponse>() {
            @Override
            public void subscribe(final ObservableEmitter<QXHttpResponse> emitter) throws Exception {
                QXHttpLogUtil.logHeader(getSimpleUrl(), getHeaders());
                QXHttpLogUtil.logParams(getSimpleUrl(), getParams());

                QXHttpSender.execute(request, new Callback() {
                    @Override
                    public void onFailure(Call call, IOException e) {
                        Request failRequest = call.request();
                        QXErrorThrowable errorThrowable = new QXErrorThrowable();
                        errorThrowable.setUrl(getSimpleUrl());
                        errorThrowable.setMethod(failRequest.method());
                        errorThrowable.setMessage(e.getMessage());
                        emitter.onError(errorThrowable);
                        emitter.onComplete();
                    }

                    @Override
                    public void onResponse(Call call, Response response) throws IOException {
                        boolean successful = response.isSuccessful();
                        int code = response.code();
                        String method = request.method();
                        QXHttpResponse httpResponse;
                        if (successful && code == 200) {
                            httpResponse = new QXHttpResponse(getSimpleUrl(), response.body().string(), true, method, code);
                        } else {
                            httpResponse = new QXHttpResponse(getSimpleUrl(), response.body().string(), false, method, code);
                        }
                        emitter.onNext(httpResponse);
                        emitter.onComplete();
                    }
                });
            }
        })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribeOn(Schedulers.io())
                .subscribe(new Observer<QXHttpResponse>() {
                    @Override
                    public void onSubscribe(Disposable d) {
                        apiCallBack.before(getSimpleUrl(), request.method());
                    }

                    @Override
                    public void onNext(QXHttpResponse response) {
                        if (response.isSuccess()) {
                            QXHttpLogUtil.log("成功", getSimpleUrl(), response.getData());
                            apiCallBack.success(response.getUrl(), response.getData(), response.getMethod());
                        } else {
                            QXHttpLogUtil.log("失败", getSimpleUrl(), "code:" + response.getCode());
                            apiCallBack.error(response.getUrl(), response.getData(), response.getMethod());
                        }
                    }

                    @Override
                    public void onError(Throwable e) {
                        QXErrorThrowable throwable = (QXErrorThrowable) e;
                        String message = throwable.getMessage();
                        String errorUrl = throwable.getUrl();
                        String errorMethod = throwable.getMethod();
                        if (SocketClosed.equals(message) || SocketIsClosed.equals(message) || Canceled.equals(message)) {
                            apiCallBack.closed(errorUrl, errorMethod);
                            QXHttpLogUtil.log("取消", getSimpleUrl(), "请求被取消");
                            apiCallBack.complete(getSimpleUrl(), errorMethod);
                            QXHttpLogUtil.log("结束", getSimpleUrl(), "接口请求结束");
                        } else {
                            apiCallBack.error(errorUrl, message, errorMethod);
                            QXHttpLogUtil.log("错误", getSimpleUrl(), e.getMessage());
                        }
                    }

                    @Override
                    public void onComplete() {
                        QXHttpLogUtil.log("完成", getSimpleUrl(), "接口请求完成");
                        apiCallBack.complete(getSimpleUrl(), request.method());
                    }
                });


    }


}