package android.library.util.http;

import android.text.TextUtils;
import okhttp3.*;

import java.io.File;
import java.util.Map;

public class QXHttpBuildUtil {
    private static final MediaType MEDIA_TYPE_ATTACH = MediaType.parse("application/octet-stream;charset=utf-8");
    private static final MediaType MEDIA_TYPE_JSON = MediaType.parse("application/json;charset=utf-8");

    /**
     * 构建一个Get Request
     *
     * @param r QXHttpBuilder
     * @return Request
     */
    public static Request createGetRequest(QXHttpBuilder r) {
        String getParams = TextUtils.isEmpty(r.getGetParams()) ? "" : "?".concat(r.getGetParams());
        Request.Builder builder = buildRequestBuilder(r.getHeaders(), r.getCacheControl(), r.getTag());
        return builder.url(r.getUrl().concat(getParams))
                .get()
                .build();
    }

    /**
     * 构建一个Post Form Request
     *
     * @param r QXHttpBuilder
     * @return Request
     */
    public static Request createPostFormRequest(QXHttpBuilder r) {
        Request.Builder builder = buildRequestBuilder(r.getHeaders(), r.getCacheControl(), r.getTag());
        return builder.url(r.getUrl())
                .post(r.getFileRequestBody())
                .build();
    }

    /**
     * 构建一个Post JSON Request
     *
     * @param r QXHttpBuilder
     * @return Request
     */
    public static Request createPostJSONRequest(QXHttpBuilder r) {
        Request.Builder builder = buildRequestBuilder(r.getHeaders(), r.getCacheControl(), r.getTag());
        return builder.url(r.getUrl())
                .post(RequestBody.create(MEDIA_TYPE_JSON, r.getJsonParams()))
                .build();
    }

    /**
     * 构建一个Delete Form Request
     *
     * @param r QXHttpBuilder
     * @return Request
     */
    public static Request createDeleteFormRequest(QXHttpBuilder r) {
        Request.Builder builder = buildRequestBuilder(r.getHeaders(), r.getCacheControl(), r.getTag());
        return builder.url(r.getUrl())
                .delete(r.getRequestBody())
                .build();
    }

    /**
     * 构建一个Delete JSON Request
     *
     * @param r QXHttpBuilder
     * @return Request
     */
    public static Request createDeleteJSONRequest(QXHttpBuilder r) {
        Request.Builder builder = buildRequestBuilder(r.getHeaders(), r.getCacheControl(), r.getTag());
        return builder.url(r.getUrl())
                .delete(RequestBody.create(MEDIA_TYPE_JSON, r.getJsonParams()))
                .build();
    }

    /**
     * 构建一个Put Form Request
     *
     * @param r QXHttpBuilder
     * @return Request
     */
    public static Request createPutFormRequest(QXHttpBuilder r) {
        Request.Builder builder = buildRequestBuilder(r.getHeaders(), r.getCacheControl(), r.getTag());
        return builder.url(r.getUrl())
                .put(r.getRequestBody())
                .build();
    }

    /**
     * 构建一个Put JSON Request
     *
     * @param r QXHttpBuilder
     * @return Request
     */
    public static Request createPutJSONRequest(QXHttpBuilder r) {
        Request.Builder builder = buildRequestBuilder(r.getHeaders(), r.getCacheControl(), r.getTag());
        return builder.url(r.getUrl())
                .put(RequestBody.create(MEDIA_TYPE_JSON, r.getJsonParams()))
                .build();
    }

    /**
     * 构建一个没有设置请求方法的 Request.Builder
     *
     * @param headers      请求头
     * @param cacheControl 缓存控制
     * @param tag          tag
     * @return Request.Builder对象
     */
    public static Request.Builder buildRequestBuilder(Headers headers, CacheControl cacheControl, Object tag) {
        Request.Builder builder = new Request.Builder();
        builder.tag(tag);
        if (headers != null) {
            builder.headers(headers);
        }
        if (cacheControl != null) {
            builder.cacheControl(cacheControl);
        }
        return builder;
    }

    /**
     * 构建一个表单
     *
     * @param map map参数集合
     * @return RequestBody
     */
    public static <K, V> RequestBody buildFormRequestBody(Map<K, V> map) {
        FormBody.Builder builder = new FormBody.Builder();
        //遍历参数
        if (map != null) {
            for (Map.Entry<K, V> entry : map.entrySet()) {
                builder.add(entry.getKey().toString(), entry.getValue().toString());
            }
        }
        return builder.build();
    }

    /**
     * 构建一个表单(带文件)
     *
     * @param map     map参数集合
     * @param fileMap map文件集合
     * @return RequestBody
     */
    public static RequestBody buildFormRequestBody(Map<String, Object> map, Map<String, File> fileMap) {
        MultipartBody.Builder builder = new MultipartBody.Builder();
        builder.setType(MultipartBody.FORM);


        //遍历参数
        if (map != null) {
            for (Map.Entry<String, Object> entry : map.entrySet()) {
                builder.addFormDataPart(entry.getKey(), entry.getValue().toString());
            }
        }
        //遍历文件
        for (Map.Entry<String, File> entry : fileMap.entrySet()) {
            File file = entry.getValue();
            if (!file.exists() || !file.isFile()) {
                continue;
            }
            builder.addFormDataPart(entry.getKey(), file.getName(), RequestBody.create(MEDIA_TYPE_ATTACH, file));
        }
        return builder.build();
    }
}
