package android.library.util.listview;

import androidx.recyclerview.widget.RecyclerView;
import android.util.SparseArray;
import android.view.View;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import java.util.Calendar;

/**
 * ListView、GridView适配器ViewHolder
 */
public class BaseListViewHolder extends RecyclerView.ViewHolder {
    private final SparseArray<View> views;
    private BaseListViewAdapter.OnChildItemViewListener onChildClickListener;
    private int position;

    public BaseListViewHolder(View itemView, BaseListViewAdapter.OnChildItemViewListener onChildClickListener) {
        super(itemView);
        views = new SparseArray<>();
        this.onChildClickListener = onChildClickListener;
    }

    /**
     * 设置view的Visibility
     *
     * @param viewId     view的id
     * @param visibility VISIBLE、INVISIBLE、GONE
     * @return BaseListViewHolder
     */
    public BaseListViewHolder setVisibility(int viewId, int visibility) {
        getView(viewId).setVisibility(visibility);
        return this;
    }

    /**
     * 设置TextView的text内容
     *
     * @param viewId  view的id
     * @param content 要显示的内容
     * @return BaseListViewHolder
     */
    public BaseListViewHolder setText(int viewId, String content) {
        getTextView(viewId).setText(content);
        return this;
    }

    /**
     * 设置TextView的字体颜色
     *
     * @param viewId view的id
     * @param color  文本的颜色
     * @return BaseListViewHolder
     */
    public BaseListViewHolder setTextColor(int viewId, int color) {
        getTextView(viewId).setTextColor(color);
        return this;
    }

    /**
     * ImageView设置resource内容
     *
     * @param viewId   view的id
     * @param resource 要设置的resource
     * @return BaseListViewHolder
     */
    public BaseListViewHolder setImage(int viewId, int resource) {
        getImageView(viewId).setImageResource(resource);
        return this;
    }

    /**
     * 给控件添加点击事件（view的唯一标识为view的id）
     *
     * @param viewId view的id
     * @return BaseListViewHolder
     */
    public BaseListViewHolder setOnClick(int viewId) {
        setOnClick(viewId, viewId);
        return this;
    }

    /**
     * 给控件添加点击事件（view的唯一标识默认为view的id）
     *
     * @param viewId view的id
     * @param id     view的唯一标识
     * @return BaseListViewHolder
     */
    public BaseListViewHolder setOnClick(int viewId, int id) {
        if (onChildClickListener != null) {
            getView(viewId).setOnClickListener(new ViewClickListener(onChildClickListener, getItemPosition(), id));
        }
        return this;
    }

    /**
     * 获取当前itemView对应的下标
     *
     * @return 当前下标
     */
    public int getItemPosition() {
        return position;
    }

    /**
     * 设置当前itemView对应的下标
     *
     * @param position 当前下标
     */
    public void setItemPosition(int position) {
        this.position = position;
    }

    /**
     * 根据view的id获取TextView
     *
     * @param viewId 布局id
     * @return TextView
     */
    public TextView getTextView(int viewId) {
        return getView(viewId);
    }

    /**
     * 根据view的id获取Button
     *
     * @param viewId 布局id
     * @return Button
     */
    public Button getButton(int viewId) {
        return getView(viewId);
    }

    /**
     * 根据view的id获取ImageView
     *
     * @param viewId 布局id
     * @return ImageView
     */
    public ImageView getImageView(int viewId) {
        return getView(viewId);
    }

    /**
     * 根据view的id获取LinearLayout
     *
     * @param viewId 布局id
     * @return LinearLayout
     */
    public LinearLayout getLinearLayout(int viewId) {
        return getView(viewId);
    }

    /**
     * 根据view的id获取RelativeLayout
     *
     * @param viewId 布局id
     * @return RelativeLayout
     */
    public RelativeLayout getRelativeLayout(int viewId) {
        return getView(viewId);
    }

    /**
     * 根据view的id获取View
     *
     * @param viewId 布局id
     * @return View
     */
    @SuppressWarnings("unchecked")
    public <T extends View> T getView(int viewId) {
        View view = views.get(viewId);
        if (view == null) {
            view = itemView.findViewById(viewId);
            views.put(viewId, view);
        }
        return (T) view;
    }

    /**
     * view的点击事件
     */
    private class ViewClickListener implements View.OnClickListener {
        private BaseListViewAdapter.OnChildItemViewListener onChildItemViewListener;
        private int position;
        private int viewType;
        public static final int MIN_CLICK_DELAY_TIME = 800;
        private long lastClickTime = 0;

        public ViewClickListener(BaseListViewAdapter.OnChildItemViewListener onViewClickListener, int position, int viewtype) {
            this.onChildItemViewListener = onViewClickListener;
            this.position = position;
            this.viewType = viewtype;
        }

        @Override
        public void onClick(View v) {
            long currentTime = Calendar.getInstance().getTimeInMillis();
            if (currentTime - lastClickTime > MIN_CLICK_DELAY_TIME) {
                lastClickTime = currentTime;
                if (onChildItemViewListener != null) {
                    onChildItemViewListener.onChildViewClick(v, position, viewType);
                }
            }
        }
    }
}