package android.library.util.listview;

import android.content.Context;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseAdapter;

import java.util.ArrayList;
import java.util.List;

/**
 * ListView、GridView适配器
 */
public abstract class BaseListViewAdapter<T> extends BaseAdapter {
    /**
     * 当前上下文
     */
    private Context context;
    /**
     * 数据源
     */
    private List<T> list;
    /**
     * itemView的布局id
     */
    private int layoutId;
    /**
     * 子控件点击事件的回调接口
     */
    private OnChildItemViewListener onChildItemViewListener;

    public BaseListViewAdapter(Context context, ArrayList<T> list, int layoutId) {
        this.context = context;
        this.list = list;
        this.layoutId = layoutId;
    }

    @Override
    public int getCount() {
        return list.size();
    }

    @Override
    public T getItem(int i) {
        return list.get(i);
    }

    @Override
    public long getItemId(int i) {
        return i;
    }

    @Override
    public View getView(int i, View view, ViewGroup viewGroup) {
        BaseListViewHolder ho = null;
        if (view == null) {
            view = onCreateView(viewGroup);
            ho = new BaseListViewHolder(view, onChildItemViewListener);
            view.setTag(ho);
        } else {
            ho = (BaseListViewHolder) view.getTag();
        }
        onBind(list.get(i), i, view, ho);
        return view;
    }

    private View onCreateView(ViewGroup parent) {
        return LayoutInflater.from(context).inflate(layoutId, parent, false);
    }

    /**
     * ListView进行数据绑定
     *
     * @param bean     当前下标对应的实体类数据
     * @param position 当前下标
     * @param itemView 当前item的布局itemView
     * @param ho       ViewHolder
     */
    public abstract void onBind(T bean, int position, View itemView, BaseListViewHolder ho);

    /**
     * item中子view点击事件传递的接口
     */
    public interface OnChildItemViewListener {
        /**
         * 点击时间回调方法
         *
         * @param itemView 当前item的布局itemView
         * @param position 当前下标
         * @param id       点击控件的唯一标识
         */
        void onChildViewClick(View itemView, int position, int id);
    }

    /**
     * 添加、设置itemView中子控件的点击事件回调
     *
     * @param onChildClickListener 回调接口
     */
    public void setOnChildClickListener(OnChildItemViewListener onChildClickListener) {
        this.onChildItemViewListener = onChildClickListener;
    }
}
