package cn.lsmya.rxkit.view.recyclerview;

import android.content.Context;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import androidx.recyclerview.widget.RecyclerView;

import java.util.List;

/**
 * recyclerView封装
 *
 * @param <T> 数据源范型
 */
public abstract class RxBaseViewAdapter<T> extends RecyclerView.Adapter<RxViewHolder> {

    private List<T> list;
    private int layoutId;
    /**
     * item点击监听
     */
    private OnItemClickListener onItemClickListener;
    /**
     * item中子View点击监听
     */
    private OnChildClickListener onChildClickListener;

    private RecyclerView recyclerView;

    public RxBaseViewAdapter(List<T> list, int layoutId) {
        this.list = list;
        this.layoutId = layoutId;
    }

    public RxBaseViewAdapter(List<T> list, int layoutId, OnChildClickListener onChildClickListener) {
        this.list = list;
        this.layoutId = layoutId;
        this.onChildClickListener = onChildClickListener;
    }

    @Override
    public RxViewHolder onCreateViewHolder(ViewGroup parent, int viewType) {
        View view = LayoutInflater.from(recyclerView.getContext()).inflate(layoutId, parent, false);
        final RxViewHolder rxViewHolder = new RxViewHolder(view, onChildClickListener);
        if (onItemClickListener != null) {
            rxViewHolder.itemView.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    onItemClickListener.onItemClick(rxViewHolder.getLayoutPosition());
                }
            });
        }
        return rxViewHolder;
    }

    @Override
    public void onBindViewHolder(RxViewHolder holder, int position) {
        onBind(list.get(position), position, holder);
    }

    @Override
    public void onAttachedToRecyclerView(RecyclerView recyclerView) {
        this.recyclerView = recyclerView;
    }

    /**
     * item数据绑定
     *
     * @param bean     当前item对应的数据
     * @param position 当前下标
     * @param ho       viewHolder
     */
    public abstract void onBind(T bean, int position, RxViewHolder ho);

    @Override
    public int getItemCount() {
        return list.size();
    }

    public Context getContext() {
        return recyclerView.getContext();
    }

    /**
     * item的点击事件
     *
     * @param onItemClickListener 回调监听
     */
    public void setOnItemClickListener(OnItemClickListener onItemClickListener) {
        this.onItemClickListener = onItemClickListener;
    }

    /**
     * item中子View的点击事件
     *
     * @param onChildClickListener 回调监听
     */
    public void setOnChildClickListener(OnChildClickListener onChildClickListener) {
        this.onChildClickListener = onChildClickListener;
    }

    /**
     * item中子控件的点击事件回调
     */
    public interface OnChildClickListener {
        void onChildClick(int position, int id);
    }

    /**
     * item点击事件回调
     */
    public interface OnItemClickListener {
        void onItemClick(int position);
    }

}
