package cn.lsmya.http.other;

import android.text.TextUtils;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.List;

import cn.lsmya.http.builder.DeleteFormBuilder;
import cn.lsmya.http.builder.DeleteJsonBuilder;
import cn.lsmya.http.builder.GetBuilder;
import cn.lsmya.http.builder.PatchFormBuilder;
import cn.lsmya.http.builder.PatchJsonBuilder;
import cn.lsmya.http.builder.PostFormBuilder;
import cn.lsmya.http.builder.PostJsonBuilder;
import cn.lsmya.http.builder.PutFormBuilder;
import cn.lsmya.http.builder.PutJsonBuilder;
import cn.lsmya.http.entity.KeyValuePair;
import io.reactivex.annotations.NonNull;
import okhttp3.CacheControl;
import okhttp3.Headers;
import okhttp3.HttpUrl;
import okhttp3.MediaType;
import okhttp3.Request;

public class BuildUtil {
    private static final MediaType MEDIA_TYPE_ATTACH = MediaType.parse("application/octet-stream;charset=utf-8");
    private static final MediaType MEDIA_TYPE_JSON = MediaType.parse("application/json;charset=utf-8");

    /**
     * 构建一个Get Request
     *
     * @param r BaseHttpBuilder
     * @return Request
     */
    public static Request createGetRequest(GetBuilder r) {
        Request.Builder builder = buildRequestBuilder(r.getHeaders(), r.getCacheControl());
        return builder.url(getHttpUrl(r.getUrl(),r.getParams()))
                .get()
                .build();
    }

    /**
     * 构建一个Post Form Request
     *
     * @param r PostFormBuilder
     * @return Request
     */
    public static Request createPostFormRequest(PostFormBuilder r) {
        Request.Builder builder = buildRequestBuilder(r.getHeaders(), r.getCacheControl());
        return builder.url(r.getUrl())
                .post(r.getRequestBody())
                .build();
    }

    /**
     * 构建一个Post JSON Request
     *
     * @param r PostJsonBuilder
     * @return Request
     */
    public static Request createPostJSONRequest(PostJsonBuilder r) {
        Request.Builder builder = buildRequestBuilder(r.getHeaders(), r.getCacheControl());
        return builder.url(r.getUrl())
                .post(r.getRequestBody())
                .build();
    }
    /**
     * 构建一个Put Form Request
     *
     * @param r PutFormBuilder
     * @return Request
     */
    public static Request createPutFormRequest(PutFormBuilder r) {
        Request.Builder builder = buildRequestBuilder(r.getHeaders(), r.getCacheControl());
        return builder.url(r.getUrl())
                .put(r.getRequestBody())
                .build();
    }

    /**
     * 构建一个Put JSON Request
     *
     * @param r PutJsonBuilder
     * @return Request
     */
    public static Request createPutJSONRequest(PutJsonBuilder r) {
        Request.Builder builder = buildRequestBuilder(r.getHeaders(), r.getCacheControl());
        return builder.url(r.getUrl())
                .put(r.getRequestBody())
                .build();
    }

    /**
     * 构建一个Delete Form Request
     *
     * @param r DeleteFormBuilder
     * @return Request
     */
    public static Request createDeleteFormRequest(DeleteFormBuilder r) {
        Request.Builder builder = buildRequestBuilder(r.getHeaders(), r.getCacheControl());
        return builder.url(r.getUrl())
                .delete(r.getRequestBody())
                .build();
    }

    /**
     * 构建一个Delete JSON Request
     *
     * @param r DeleteJsonBuilder
     * @return Request
     */
    public static Request createDeleteJSONRequest(DeleteJsonBuilder r) {
        Request.Builder builder = buildRequestBuilder(r.getHeaders(), r.getCacheControl());
        return builder.url(r.getUrl())
                .delete(r.getRequestBody())
                .build();
    }

    /**
     * 构建一个Patch Form Request
     *
     * @param r DeleteFormBuilder
     * @return Request
     */
    public static Request createPatchFormRequest(PatchFormBuilder r) {
        Request.Builder builder = buildRequestBuilder(r.getHeaders(), r.getCacheControl());
        return builder.url(r.getUrl())
                .patch(r.getRequestBody())
                .build();
    }

    /**
     * 构建一个Patch JSON Request
     *
     * @param r DeleteJsonBuilder
     * @return Request
     */
    public static Request createPatchJSONRequest(PatchJsonBuilder r) {
        Request.Builder builder = buildRequestBuilder(r.getHeaders(), r.getCacheControl());
        return builder.url(r.getUrl())
                .patch(r.getRequestBody())
                .build();
    }





    /**
     * 构建一个没有设置请求方法的 Request.Builder
     *
     * @param headers      请求头
     * @param cacheControl 缓存控制
     * @return Request.Builder对象
     */
    public static Request.Builder buildRequestBuilder(Headers headers, CacheControl cacheControl) {
        Request.Builder builder = new Request.Builder();
        if (headers != null) {
            builder.headers(headers);
        }
        if (cacheControl != null) {
            builder.cacheControl(cacheControl);
        }
        return builder;
    }

    /**
     * 对中文进行编码
     * @param fileName
     * @return
     */
    private static String getValidZH(String fileName) {
        if (TextUtils.isEmpty(fileName)) {
            return String.valueOf(System.currentTimeMillis());
        }
        String validUA = String.valueOf(System.currentTimeMillis());
        String uaWithoutLine = fileName.replace("\n", "");
        for (int i = 0, length = uaWithoutLine.length(); i < length; i++) {
            char c = fileName.charAt(i);
            if (c <= '\u001f' || c >= '\u007f') {
                try {
                    validUA = URLEncoder.encode(uaWithoutLine, "UTF-8");
                } catch (UnsupportedEncodingException e) {
                    e.printStackTrace();
                }
                return validUA;
            }
        }
        return uaWithoutLine;
    }

    public static HttpUrl getHttpUrl(@NonNull String url, List<KeyValuePair> list) {
        HttpUrl httpUrl = HttpUrl.get(url);
        if (list == null || list.size() == 0) {
            return httpUrl;
        }
        HttpUrl.Builder builder = httpUrl.newBuilder();
        for (KeyValuePair pair : list) {
            if (pair.isEncoded()) {
                builder.addEncodedQueryParameter(pair.getKey(), pair.getValue().toString());
            } else {
                builder.addQueryParameter(pair.getKey(), pair.getValue().toString());
            }
        }
        return builder.build();
    }


}
