package cn.lsmya.http.builder;

import android.text.TextUtils;
import android.util.Patterns;

import java.io.IOException;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import cn.lsmya.http.GsonUtil;
import cn.lsmya.http.HttpManage;
import cn.lsmya.http.HttpStatusCodeException;
import cn.lsmya.http.IOUtil;
import cn.lsmya.http.ParameterizedTypeImpl;
import cn.lsmya.http.entity.KeyValuePair;
import cn.lsmya.http.listener.OnProgressListener;
import cn.lsmya.http.other.BuildUtil;
import cn.lsmya.http.other.HttpLogUtil;
import cn.lsmya.http.other.HttpSender;
import cn.lsmya.http.param.IHeaders;
import cn.lsmya.http.param.IMethod;
import cn.lsmya.http.param.IObservable;
import cn.lsmya.http.param.IParam;
import io.reactivex.Observable;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.schedulers.Schedulers;
import okhttp3.CacheControl;
import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Headers;
import okhttp3.Headers.Builder;
import okhttp3.Request;
import okhttp3.Response;

public class HttpBuilder<B> implements IHeaders<B>, IParam<B>, IMethod<B>, IObservable {

    /**
     * 接口地址
     */
    private String mUrl;
    private CacheControl mCacheControl;

    /**
     * 请求参数集合
     */
    private List<KeyValuePair> paramsList;

    /**
     * 请求头构造器
     */
    private Builder mHBuilder;

    public HttpBuilder(String url) {
        mUrl = url;
    }

    /****************************请求头****************************/
    @Override
    public B addHeader(String line) {
        return addHeader(line, true);
    }

    @Override
    public B addHeader(String line, boolean isAdd) {
        if (isAdd) {
            getHeadersBuilder().add(line);
        }
        return (B) this;
    }

    @Override
    public B addHeader(String key, String value) {
        return addHeader(key, value, true);
    }

    @Override
    public B addHeader(String key, String value, boolean isAdd) {
        if (isAdd && !TextUtils.isEmpty(key) && !TextUtils.isEmpty(value)) {
            getHeadersBuilder().add(key, value);
        }
        return (B) this;
    }

    @Override
    public B addHeader(Map<String, String> headerMap) {
        return addHeader(headerMap, true);
    }

    @Override
    public B addHeader(Map<String, String> headerMap, boolean isAdd) {
        if (isAdd && headerMap != null) {
            for (Map.Entry<String, String> entry : headerMap.entrySet()) {
                String key = entry.getKey();
                String value = entry.getValue();
                addHeader(key, value);
            }
        }
        return (B) this;
    }

    @Override
    public B setHeader(String key, String value) {
        return setHeader(key, value, true);
    }

    @Override
    public B setHeader(String key, String value, boolean isAdd) {
        if (isAdd) {
            getHeadersBuilder().set(key, value);
        }
        return (B) this;
    }

    @Override
    public String getHeader(String key) {
        return getHeadersBuilder().get(key);
    }

    @Override
    public Headers getHeaders() {
        if (mHBuilder == null) {
            return null;
        } else {
            return mHBuilder.build();
        }
    }

    @Override
    public Builder getHeadersBuilder() {
        if (mHBuilder == null) {
            mHBuilder = new Builder();
        }
        return mHBuilder;
    }

    @Override
    public B setHeadersBuilder(Builder builder) {
        mHBuilder = builder;
        return (B) this;
    }

    @Override
    public B removeAllHeader(String key) {
        getHeadersBuilder().removeAll(key);
        return (B) this;
    }

    /****************************请求url、方法****************************/
    @Override
    public B setUrl(String url) {
        this.mUrl = url;
        return (B) this;
    }

    @Override
    public String getUrl() {
        if (TextUtils.isEmpty(HttpManage.getBaseUrl())) {
            throw new IllegalArgumentException("baseUrl cannot be empty");
        } else {
            if (mUrl == null) {
                mUrl = "";
            }
            if (Patterns.WEB_URL.matcher(mUrl).matches()) {
                return mUrl;
            } else {
                if (HttpManage.getBaseUrl().endsWith("/")) {
                    if (mUrl.startsWith("/")) {
                        return HttpManage.getBaseUrl() + mUrl.substring(1);
                    } else {
                        return HttpManage.getBaseUrl() + mUrl;
                    }
                } else {
                    if (mUrl.startsWith("/")) {
                        return HttpManage.getBaseUrl() + mUrl;
                    } else {
                        return HttpManage.getBaseUrl() + "/" + mUrl;
                    }
                }
            }
        }
    }

    @Override
    public CacheControl getCacheControl() {
        return mCacheControl;
    }

    @Override
    public B cacheControl(CacheControl cacheControl) {
        mCacheControl = cacheControl;
        return (B) this;
    }

    /****************************请求参数****************************/

    @Override
    public B add(String key, Object value) {
        return add(key, value, true);
    }

    @Override
    public B add(String key, Object value, boolean isAdd) {
        if (isAdd) {
            getParams().add(new KeyValuePair(key, value));
        }
        return (B) this;
    }

    @Override
    public B add(Map<String, Object> map) {
        return add(map, true);
    }

    @Override
    public B add(Map<String, Object> map, boolean isAdd) {
        if (isAdd && map != null) {
            for (Map.Entry<String, Object> entry : map.entrySet()) {
                getParams().add(new KeyValuePair(entry.getKey(), entry.getValue()));
            }
        }
        return (B) this;
    }

    @Override
    public List<KeyValuePair> getParams() {
        if (paramsList == null) {
            paramsList = new ArrayList<>();
        }
        return paramsList;
    }

    /****************************请求结果处理****************************/

    @Override
    public Observable<String> asString() {
        return asObject(String.class);
    }

    @Override
    public <T> Observable<List<T>> asArray(Class<T> cls) {
        return request((B) this, new ParameterizedTypeImpl(ArrayList.class, new Class[]{cls}));
    }

    @Override
    public <T> Observable<T> asObject(Class<T> cls) {
        return request((B) this, cls);
    }

    @Override
    public Observable<String> asDownload(String path) {
        return asDownload(path,null);
    }
    @Override
    public Observable<String> asDownload(String path,OnProgressListener onProgressListener) {
        return requestDownload(onProgressListener,(B) this, path);
    }

    private <T> Observable<T> request(B builder, final Type type) {
        final Request request = HttpManage.getRequest(getRequest(builder));
        return Observable.create((ObservableOnSubscribe<T>) emitter -> {
            HttpLogUtil.log(request);
            HttpSender.execute(request, new Callback() {
                @Override
                public void onFailure(Call call, IOException e) {
                    emitter.onError(e);
                    emitter.onComplete();
                }

                @Override
                public void onResponse(Call call, Response response) throws IOException {
                    HttpLogUtil.log(response, null);
                    boolean successful = response.isSuccessful();
                    int code = response.code();
                    if (successful && code == 200) {
                        if (type.equals(String.class)){
                            emitter.onNext((T) response.body().string());
                        }else {
                            emitter.onNext(GsonUtil.fromJson(response.body().string(), type));
                        }
                    } else {
                        emitter.onError(new HttpStatusCodeException(response));
                    }
                    emitter.onComplete();
                }
            });
        })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribeOn(Schedulers.io());
    }

    private Observable<String> requestDownload(final OnProgressListener listener, B builder, final String path) {
        final Request request = HttpManage.getRequest(getRequest(builder));
        return Observable.create((ObservableOnSubscribe<String>) emitter -> {
            HttpLogUtil.log(request);
            HttpSender.download(listener,request, new Callback() {
                @Override
                public void onFailure(Call call, IOException e) {
                    emitter.onError(e);
                    emitter.onComplete();
                }

                @Override
                public void onResponse(Call call, Response response) throws IOException {
                    HttpLogUtil.log(response, null);
                    boolean successful = response.isSuccessful();
                    int code = response.code();
                    if (successful && code == 200) {
                        boolean append = response.header("Content-Range") != null;
                        //将输入流写出到文件
                        IOUtil.write(response.body().byteStream(), path, append);
                        emitter.onNext(path);
                    } else {
                        emitter.onError(new HttpStatusCodeException(response));
                    }
                    emitter.onComplete();
                }
            });
        })
                .observeOn(AndroidSchedulers.mainThread())
                .subscribeOn(Schedulers.io());
    }

    private Request getRequest(B builder) {
        if (builder instanceof GetBuilder) {
            return BuildUtil.createGetRequest((GetBuilder) builder);
        } else if (builder instanceof PostFormBuilder) {
            return BuildUtil.createPostFormRequest((PostFormBuilder) builder);
        } else if (builder instanceof PostJsonBuilder) {
            return BuildUtil.createPostJSONRequest((PostJsonBuilder) builder);
        } else if (builder instanceof PutFormBuilder) {
            return BuildUtil.createPutFormRequest((PutFormBuilder) builder);
        } else if (builder instanceof PutJsonBuilder) {
            return BuildUtil.createPutJSONRequest((PutJsonBuilder) builder);
        } else if (builder instanceof DeleteFormBuilder) {
            return BuildUtil.createDeleteFormRequest((DeleteFormBuilder) builder);
        } else if (builder instanceof DeleteJsonBuilder) {
            return BuildUtil.createDeleteJSONRequest((DeleteJsonBuilder) builder);
        } else if (builder instanceof PatchFormBuilder) {
            return BuildUtil.createPatchFormRequest((PatchFormBuilder) builder);
        } else if (builder instanceof PatchJsonBuilder) {
            return BuildUtil.createPatchJSONRequest((PatchJsonBuilder) builder);
        }
        return null;
    }
}