package rx.library.dialog;

import android.app.Dialog;
import android.content.Context;
import android.content.res.ColorStateList;
import android.graphics.Color;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.GradientDrawable;
import android.os.Build;
import android.text.TextUtils;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.view.WindowManager;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.appcompat.content.res.AppCompatResources;
import androidx.core.content.ContextCompat;

import java.util.ArrayList;
import java.util.List;

import rx.library.dialog.interfaces.OnDialogClickListener;

public class RxDialog {
    public static final int HORIZONTAL = 0;
    public static final int CANCEL = 2;

    public static Builder newBuilder(Context context) {
        return new Builder(context);
    }

    public static class Builder {
        private Context context;
        private String mTitle;
        private String mMessage;
        private int mGravity = Gravity.CENTER;
        private float mWidth = 0.7f;
        private List<ButtonView> list;
        private int mStyle = HORIZONTAL;
        private int backgroundColor;

        public Builder(Context context) {
            this.context = context;
            backgroundColor = ContextCompat.getColor(context, R.color.kit_dialogBackground);
        }

        public Builder setTitle(String title) {
            this.mTitle = title;
            return this;
        }

        public Builder setMessage(String message) {
            this.mMessage = message;
            return this;
        }

        public Builder addButton(String butString) {
            addButton(butString, null);
            return this;
        }

        public Builder addButton(String butString, int textColor) {
            addButton(butString, textColor, null);
            return this;
        }

        public Builder addButton(String butString, OnDialogClickListener listener) {
            addButton(butString, R.color.kit_dialog_textview, listener);
            return this;
        }

        public Builder setBackgroundColor(int color) {
            backgroundColor = color;
            return this;
        }

        public Builder addButton(String butString, int textColor, final OnDialogClickListener listener) {
            if (list == null) {
                list = new ArrayList<>();
            }
            TextView textView = new TextView(context);
            textView.setText(butString);
            textView.setClickable(true);
            textView.setTextSize(16);
            ColorStateList colorStateList = AppCompatResources.getColorStateList(context, textColor);
            textView.setTextColor(colorStateList);
            textView.setGravity(Gravity.CENTER);
            ButtonView buttonView = new ButtonView();
            buttonView.setTextView(textView);
            buttonView.setListener(listener);
            list.add(buttonView);
            return this;
        }

        public Builder setWidth(float width) {
            this.mWidth = width;
            return this;
        }

        public Builder setGravity(int gravity) {
            this.mGravity = gravity;
            return this;
        }

        public Builder setStyle(int style) {
            mStyle = style;
            return this;
        }

        public Dialog build() {
            final Dialog dialog = new Dialog(context);
            View view = LayoutInflater.from(context).inflate(R.layout.kit_dialog, null);
            TextView cancelView = view.findViewById(R.id.kit_ui_dialog_cancel);
            //标题
            TextView titleView = view.findViewById(R.id.kit_ui_dialog_title);
            if (!TextUtils.isEmpty(mTitle)) {
                titleView.setText(mTitle);
                titleView.setBackgroundColor(backgroundColor);
            } else {
                if (mStyle != HORIZONTAL) {
                    titleView.setVisibility(View.GONE);
                }
            }
            //提示内容
            TextView messageView = view.findViewById(R.id.kit_ui_dialog_message);
            if (!TextUtils.isEmpty(mMessage)) {
                messageView.setText(mMessage);
                messageView.setBackgroundColor(backgroundColor);
            } else {
                messageView.setVisibility(View.GONE);
            }
            if (titleView.getVisibility() == View.GONE && messageView.getVisibility() == View.GONE) {
                view.findViewById(R.id.kit_ui_dialog_division).setVisibility(View.GONE);
            }
            //按钮
            final LinearLayout linearLayout = view.findViewById(R.id.kit_ui_dialog_contentLayout);
            if (mStyle == HORIZONTAL) {
                linearLayout.setOrientation(LinearLayout.HORIZONTAL);
                LinearLayout.LayoutParams lp = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT);
                lp.height = (int) context.getResources().getDimension(R.dimen.dp45);
                linearLayout.setLayoutParams(lp);
            } else {
                linearLayout.setOrientation(LinearLayout.VERTICAL);
                LinearLayout.LayoutParams lp = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT);
                linearLayout.setLayoutParams(lp);
            }
            for (int i = 0; i < list.size(); i++) {
                if (i != 0) {
                    View view1 = new View(context);
                    view1.setBackgroundColor(ContextCompat.getColor(context, R.color.kit_divider));
                    linearLayout.addView(view1);
                    if (mStyle == HORIZONTAL) {
                        LinearLayout.LayoutParams llp = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);
                        llp.width = 2;
                        view1.setLayoutParams(llp);
                    } else {
                        LinearLayout.LayoutParams llp = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);
                        llp.height = 2;
                        view1.setLayoutParams(llp);
                    }
                }
                final View v = list.get(i).getTextView();
                if (mStyle == HORIZONTAL) {
                    LinearLayout.LayoutParams lp = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);
                    lp.weight = 1;
                    v.setLayoutParams(lp);
                } else {
                    LinearLayout.LayoutParams lp = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT);
                    lp.height = (int) context.getResources().getDimension(R.dimen.dp45);
                    v.setLayoutParams(lp);
                }
                final OnDialogClickListener listener = list.get(i).getListener();
                v.setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        dialog.cancel();
                        if (listener != null) {
                            listener.onClick();
                        }
                    }
                });
                linearLayout.addView(v);
            }

            if (mStyle == CANCEL) {
                cancelView.setVisibility(View.VISIBLE);
                cancelView.setClickable(true);
                cancelView.setTextSize(16);
                cancelView.setBackgroundColor(backgroundColor);
                ColorStateList colorStateList = AppCompatResources.getColorStateList(context, R.color.kit_dialog_textview_red);
                cancelView.setTextColor(colorStateList);
                cancelView.setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        dialog.cancel();
                    }
                });
            } else {
                cancelView.setVisibility(View.GONE);
            }

            if (mWidth >= 1.0f) {
                view.findViewById(R.id.kit_ui_dialog_layout).setBackgroundColor(Color.WHITE);
                GradientDrawable drawable = new GradientDrawable();
                drawable.setCornerRadius(0);
                drawable.setColor(backgroundColor);
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                    cancelView.setBackground(drawable);
                } else {
                    cancelView.setBackgroundDrawable(drawable);
                }
            } else {
                GradientDrawable drawable = new GradientDrawable();
                drawable.setCornerRadius(context.getResources().getDimension(R.dimen.dp15));
                drawable.setColor(backgroundColor);
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                    view.findViewById(R.id.kit_ui_dialog_layout).setBackground(drawable);
                    cancelView.setBackground(drawable);
                } else {
                    cancelView.setBackgroundDrawable(drawable);
                }
            }
            dialog.setContentView(view);
            Window window = dialog.getWindow();
            assert window != null;

            WindowManager.LayoutParams lp = window.getAttributes();
            window.setBackgroundDrawable(new ColorDrawable(Color.TRANSPARENT));
            if (mWidth >= 1.0f) {
                lp.width = context.getResources().getDisplayMetrics().widthPixels;
            } else {
                lp.width = (int) (context.getResources().getDisplayMetrics().widthPixels * mWidth);
            }
            dialog.getWindow().setAttributes(lp);
            window.setGravity(mGravity);
            dialog.show();
            return dialog;
        }
    }

    private static class ButtonView {
        private TextView textView;
        private OnDialogClickListener listener;

        public TextView getTextView() {
            return textView;
        }

        public void setTextView(TextView textView) {
            this.textView = textView;
        }

        public OnDialogClickListener getListener() {
            return listener;
        }

        public void setListener(OnDialogClickListener listener) {
            this.listener = listener;
        }
    }
}
