package cn.lsmya.dialog;

import android.app.Activity;
import android.app.Dialog;
import android.content.Context;
import android.graphics.drawable.ColorDrawable;
import android.os.Bundle;
import android.util.DisplayMetrics;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.view.WindowManager;
import android.view.inputmethod.InputMethodManager;
import android.widget.AdapterView;
import android.widget.BaseAdapter;
import android.widget.ListView;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;

import java.util.List;

/**
 * 列表dialog弹窗，宽度为屏幕宽度十分之九
 * 实例化时传入上下文以及数据源
 * setOnListItemClickListener()方法实现对列表的点击事件监听
 */
public class ChooseListDialog<T> extends Dialog implements AdapterView.OnItemClickListener {

    private Context context;
    private List<T> list;

    private ListDialogAdapter adapter;
    private OnBindDataListener onBindDataListener;
    private OnListItemClickListener listener;
    private int mGravity = Gravity.CENTER;
    private float mWidth = 1f;

    private ChooseListDialog(@NonNull Context context) {
        super(context);
        this.context = context;
    }

    private ChooseListDialog(@NonNull Context context, int themeResId) {
        super(context, themeResId);
    }

    private ChooseListDialog(@NonNull Context context, boolean cancelable, @Nullable OnCancelListener cancelListener) {
        super(context, cancelable, cancelListener);
    }

    public static ChooseListDialog getInstance(Context context) {
        return new ChooseListDialog(context);
    }

    public ChooseListDialog setData(List<T> list) {
        this.list = list;
        return this;
    }

    public ChooseListDialog setGravity(int gravity) {
        this.mGravity = gravity;
        return this;
    }

    public ChooseListDialog setWidth(float width) {
        this.mWidth = width;
        return this;
    }

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        Window window = getWindow();
        window.setGravity(mGravity);

        ListView listView = new ListView(getContext());
        listView.setDivider(new ColorDrawable(ContextCompat.getColor(context, R.color.helper_dialog_dividerColor)));
        listView.setDividerHeight(1);
        adapter = new ListDialogAdapter(list);
        listView.setAdapter(adapter);
        listView.setOnItemClickListener(this);
        setContentView(listView);
        WindowManager manager = ((Activity) context).getWindowManager();
        DisplayMetrics outMetrics = new DisplayMetrics();
        manager.getDefaultDisplay().getMetrics(outMetrics);
        WindowManager.LayoutParams lp = getWindow().getAttributes();
        if (mWidth == 1) {
            lp.width = outMetrics.widthPixels;
        } else {
            lp.width = (int) (outMetrics.widthPixels * mWidth);
        }
        getWindow().setAttributes(lp);
        setCanceledOnTouchOutside(true);
    }

    @Override
    public void onItemClick(AdapterView<?> parent, View view, int position, long id) {
        if (listener != null) {
            cancel();
            listener.onItemClick(list.get(position), position);
        }
    }


    public interface OnBindDataListener<T> {
        void onBinding(TextView view, T t);
    }

    public ChooseListDialog setOnBindDataListener(OnBindDataListener listener) {
        this.onBindDataListener = listener;
        return this;
    }

    public interface OnListItemClickListener<T> {
        void onItemClick(T t, int position);
    }

    public ChooseListDialog show(OnListItemClickListener<T> onItemClickListener) {
        this.listener = onItemClickListener;
        show();
        return this;
    }

    private class ListDialogAdapter extends BaseAdapter {
        private List<T> mList;

        public ListDialogAdapter(List<T> mList) {
            this.mList = mList;
        }

        @Override
        public int getCount() {
            return mList.size();
        }

        @Override
        public T getItem(int position) {
            return mList.get(position);
        }

        @Override
        public long getItemId(int position) {
            return position;
        }

        @Override
        public View getView(int position, View convertView, ViewGroup parent) {
            ViewHolder ho = null;
            if (convertView == null) {
                convertView = LayoutInflater.from(getContext()).inflate(R.layout.helper_dialog_item_list, null);
                ho = new ViewHolder(convertView);
                convertView.setTag(ho);
            } else {
                ho = (ViewHolder) convertView.getTag();
            }

            onBindData(ho.textView, getItem(position));
            return convertView;
        }

        class ViewHolder {
            private TextView textView;

            public ViewHolder(@NonNull View itemView) {
                textView = itemView.findViewById(R.id.helper_dialog_item_list_text);
            }
        }
    }

    private void onBindData(TextView view, T t) {
        if (onBindDataListener != null) {
            onBindDataListener.onBinding(view, t);
        }
    }

    @Override
    public void dismiss() {
        View view = getCurrentFocus();
        if (view instanceof TextView) {
            InputMethodManager mInputMethodManager = (InputMethodManager) getContext().getSystemService(Context.INPUT_METHOD_SERVICE);
            mInputMethodManager.hideSoftInputFromWindow(view.getWindowToken(), InputMethodManager.RESULT_UNCHANGED_SHOWN);
        }
        super.dismiss();
    }
}
