package cn.lsmya.view

import android.content.Context
import android.graphics.Canvas
import android.graphics.ColorFilter
import android.graphics.Paint
import android.graphics.PixelFormat
import android.graphics.drawable.Drawable
import android.util.AttributeSet
import android.view.Gravity
import android.view.MotionEvent
import android.view.View
import android.view.ViewGroup
import android.widget.LinearLayout
import android.widget.ScrollView
import android.widget.TextView
import androidx.core.content.ContextCompat

/**
 * Picker选择器
 */
class UIPickerView<T> @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
) :
    ScrollView(context, attrs, defStyleAttr) {
    /**
     * 数据源
     */
    private var mData: ArrayList<T> = ArrayList()

    /**
     * 两条线的画笔
     */
    private var mLinePaint: Paint? = null

    /**
     * 宽,高,还有显示的TextView的高度（注意:这里只是TextView的高度,不算margin）
     */
    private var mWidth = 0
    private var mHeight = 0
    private var mTextHeight: Int

    /**
     * TextView的父容器
     */
    private var mTextGroup: LinearLayout? = null

    /**
     * 获取屏幕的密度
     */
    private val mDensity: Float

    /**
     * 字体的大小
     */
    private var mTextSize = 18f

    /**
     * 根据字体大小来控制整个控件的高度
     */
    private var mWrapContentHeight: Float

    /**
     * 当前选中的position
     */
    private var position = 0
    private var tempPosition = -1

    private var mScrollY = 0

    private var mOnSetTextListener: OnSetTextListener<T>? = null
    private var mOnItemSelectListener: ((list: ArrayList<T>, position: Int) -> Unit)? = null
    private fun dp2px(dip: Int): Int {
        val density = context.resources.displayMetrics.density
        return (dip * density + 0.5f).toInt()
    }

    /**
     * 设置字体大小
     */
    fun setTextSize(textSize: Int) {
        mTextSize = textSize.toFloat()
        mTextHeight = ((mDensity + 0.5) * mTextSize).toInt()
        /**
         * 整个控件的高度为6个TextView的高度
         */
        mWrapContentHeight = mTextHeight * 6.toFloat()
    }

    private fun init() {
        try {
            removeAllViews()
        } catch (e: Exception) {
            e.printStackTrace()
        }
        /**
         * 初始化数据,首先添加Group
         */
        mTextGroup = LinearLayout(context)
        mTextGroup!!.orientation = LinearLayout.VERTICAL
        mTextGroup!!.gravity = Gravity.CENTER
        addView(mTextGroup)
        /**
         * 由于我们需要给自身的数据在选中的框里显示
         * 所以这里需要添加前面和后面的空数据
         */
        mTextGroup!!.addView(createTextView(null, false))
        for (i in mData.indices) {
            mTextGroup!!.addView(createTextView(mData[i], true))
        }
        mTextGroup!!.addView(createTextView(null, false))
        /**
         * 设置背景,这里选择画一个,两条线
         */
        background = object : Drawable() {
            override fun draw(canvas: Canvas) {
                /**
                 * 这里把两条线之间的距离设置为了两个TextView的高度
                 */
                canvas.drawLine(
                    mWidth * 0.1f,
                    mHeight / 2 - mTextHeight.toFloat(),
                    mWidth * 0.9f,
                    mHeight / 2 - mTextHeight.toFloat(),
                    mLinePaint!!
                )
                canvas.drawLine(
                    mWidth * 0.1f,
                    mHeight / 2 + mTextHeight.toFloat(),
                    mWidth * 0.9f,
                    mHeight / 2 + mTextHeight.toFloat(),
                    mLinePaint!!
                )
            }

            override fun setAlpha(i: Int) {}
            override fun setColorFilter(colorFilter: ColorFilter?) {}
            override fun getOpacity(): Int {
                return PixelFormat.UNKNOWN
            }
        }
        mLinePaint = Paint()
        mLinePaint!!.isAntiAlias = true
        mLinePaint!!.strokeWidth = 1f
        mLinePaint!!.color = ContextCompat.getColor(context, android.R.color.darker_gray)
        setPosition(0)
    }

    /**
     * 动态创建TextView
     */
    private fun createTextView(item: T?, custom: Boolean): TextView {
        val tv = TextView(context)
        if (custom) {
            if (mOnSetTextListener != null) {
                mOnSetTextListener!!.onSet(tv, item)
            }
        } else {
            tv.text = ""
        }
        tv.setTextColor(ContextCompat.getColor(context, android.R.color.darker_gray))
        tv.textSize = mTextSize
        tv.gravity = Gravity.CENTER
        val params =
            LinearLayout.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT, mTextHeight)
        params.bottomMargin = mTextHeight / 2
        params.topMargin = mTextHeight / 2
        tv.layoutParams = params
        return tv
    }

    override fun onMeasure(widthMeasureSpec: Int, heightMeasureSpec: Int) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec)
        /**
         * 更改测量方法
         * 指定高度为mWrapContentHeight
         */
        mWidth = measureWidth(widthMeasureSpec)
        mHeight = mWrapContentHeight.toInt()
        setMeasuredDimension(mWidth, mWrapContentHeight.toInt())
    }

    private fun measureWidth(size: Int): Int {
        val mode = MeasureSpec.getMode(size)
        return when (mode) {
            MeasureSpec.UNSPECIFIED, MeasureSpec.AT_MOST -> mWrapContentHeight.toInt()
            MeasureSpec.EXACTLY -> MeasureSpec.getSize(size)
            else -> MeasureSpec.getSize(size)
        }
    }

    /**
     * ScrollView的滑动事件监听
     */
    override fun onScrollChanged(l: Int, t: Int, oldl: Int, oldt: Int) {
        super.onScrollChanged(l, t, oldl, oldt)
        /**
         * 计算出当前在两条线里的position
         */
        position = (t + mTextHeight) / (mTextHeight * 2)
        /**
         * 因为此方法会在滑动的时候不停的调用,所以这里设置一个临时的变量来控制
         */
        if (tempPosition != position) {
            val size = mTextGroup!!.childCount
            for (i in 0 until size) {
                val tv = mTextGroup!!.getChildAt(i) as TextView
                /**
                 * 因为我们在数据开头添加了一个空的数据,所以这里position要+1
                 */
                if (position + 1 == i) {
                    tv.setTextColor(ContextCompat.getColor(context, android.R.color.black))
                } else {
                    tv.setTextColor(
                        ContextCompat.getColor(
                            context,
                            android.R.color.darker_gray
                        )
                    )
                }
            }
        }
        tempPosition = position
    }

    override fun onTouchEvent(ev: MotionEvent): Boolean {
        /**
         * 因为ScrollView没有停止滑动的监听,所以这里取巧
         * 在手指离开屏幕的30ms后判断是否和原来的scrollY一样
         * 如果一样则进入,如果不一样则设置为一样
         */
        if (ev.action == MotionEvent.ACTION_UP) {
            mScrollY = getScrollY()
            postDelayed(object : Runnable {
                override fun run() {
                    if (mScrollY == getScrollY()) {
                        /**
                         * 获得每次松手后scrollY相对于TextView高度的偏移量
                         */
                        val offset = mScrollY % (mTextHeight * 2)
                        /**
                         * 如果偏移量大于TextView高度的一半
                         * 则进入到下一个
                         */
                        if (offset > mTextHeight) {
                            smoothScrollTo(0, mScrollY - offset + mTextHeight * 2)
                        } else {
                            smoothScrollTo(0, mScrollY - offset)
                        }
                    } else {
                        mScrollY = getScrollY()
                        post(this)
                    }
                    mOnItemSelectListener?.invoke(mData, position)
                }
            }, 30)
        }
        return super.onTouchEvent(ev)
    }

    /**
     * 设置fling的速度为原来的1/3
     */
    override fun fling(velocityY: Int) {
        super.fling(velocityY / 3)
    }

    /**
     * 设置数据源
     */
    fun setData(data: ArrayList<T> = ArrayList()) {
        mData.clear()
        mData.addAll(data)
        init()
    }

    fun setOnSetTextListener(onSetTextListener: OnSetTextListener<T>?) {
        mOnSetTextListener = onSetTextListener
    }

    /**
     * 设置选择item回调监听
     */
    fun setOnItemSelectListener(onItemSelect: ((list: ArrayList<T>, position: Int) -> Unit)) {
        mOnItemSelectListener = onItemSelect
    }

    /**
     * 设置position
     */
    fun setPosition(position: Int) {
        this.position = position
        if (position == 0) {
            post {
                scrollTo(0, 1)
                try {
                    val size = mTextGroup!!.childCount
                    for (i in 0 until size) {
                        val tv = mTextGroup!!.getChildAt(i) as TextView
                        /**
                         * 因为我们在数据开头添加了一个空的数据,所以这里position要+1
                         */
                        /**
                         * 因为我们在数据开头添加了一个空的数据,所以这里position要+1
                         */
                        if (position + 1 == i) {
                            tv.setTextColor(
                                ContextCompat.getColor(
                                    context,
                                    android.R.color.black
                                )
                            )
                        } else {
                            tv.setTextColor(
                                ContextCompat.getColor(
                                    context,
                                    android.R.color.darker_gray
                                )
                            )
                        }
                    }
                } catch (e: Exception) {
                }
                mOnItemSelectListener?.invoke(mData, position)
            }
            return
        }
        post {
            scrollTo(0, this@UIPickerView.position * (mTextHeight * 2))
            mOnItemSelectListener?.invoke(mData, position)
        }
    }

    /**
     * 获取position
     *
     * @return 当前选择下标
     */
    fun getPosition(): Int {
        return position
    }

    /**
     * 获取当前选择的内容
     *
     * @return 当前选择的内容
     */
    val selectData: T?
        get() = try {
            mData[getPosition()]
        } catch (e: Exception) {
            null
        }

    /**
     * 设置文本内容回调接口
     */
    interface OnSetTextListener<T> {
        fun onSet(textView: TextView?, item: T?)
    }

    init {
        /**
         * 这句话设置ScrollView滑动到极限的时候不显示提示（就是那个阴影）
         */
        overScrollMode = View.OVER_SCROLL_NEVER
        /**
         * 获取到屏幕的密度来设置TextView的高度
         */
        mDensity = resources.displayMetrics.density
        mTextHeight = ((mDensity + 0.5) * mTextSize).toInt()
        /**
         * 整个控件的高度为6个TextView的高度
         */
        mWrapContentHeight = mTextHeight * 6.toFloat()
        val typedArray = context.obtainStyledAttributes(attrs, R.styleable.UIPickerView)
        for (i in 0 until typedArray.indexCount) {
            val attr = typedArray.getIndex(i)
            if (attr == R.styleable.UIPickerView_ui_textSize) {
                val textSize =
                    typedArray.getDimensionPixelSize(attr, dp2px(18)).toFloat()
                val fontScale = resources.displayMetrics.scaledDensity
                setTextSize((textSize / fontScale + 0.5f).toInt())
            }
        }
        typedArray.recycle()
    }
}