package cn.lsmya.view.progress;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.LinearGradient;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffXfermode;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.Shader;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.View;

import androidx.annotation.Nullable;

import cn.lsmya.view.R;

public class UIProgressView extends View {

    private float maxProgress = 100;
    private float minProgress = 0;

    private int hintColor = 0xFFD7D7D7;
    private int progressColor = 0xFFFFFFFF;
    private int progressStartColor = -1;
    private int progressStopColor = -1;

    private Paint paint;
    private Paint mHintPaint = new Paint();
    private Paint mPaint = new Paint();

    private int progress = 0;
    private Canvas canvas2;
    private Bitmap bitmap;
    private OnProgressChangeListener mListener;

    public UIProgressView(Context context) {
        this(context, null);
    }

    public UIProgressView(Context context, @Nullable AttributeSet attrs) {
        this(context, attrs, 0);
        init(context, attrs);
    }

    public UIProgressView(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init(context, attrs);
    }

    private void init(Context context, AttributeSet attrs) {
        TypedArray typedArray = context.obtainStyledAttributes(attrs, R.styleable.UIProgressView);
        for (int i = 0; i < typedArray.getIndexCount(); i++) {
            int attr = typedArray.getIndex(i);
            if (attr == R.styleable.UIProgressView_ui_hintColor) {
                hintColor = typedArray.getColor(attr, 0xFFD7D7D7);
            } else if (attr == R.styleable.UIProgressView_ui_progressColor) {
                progressColor = typedArray.getColor(attr, 0xFF000000);
            } else if (attr == R.styleable.UIProgressView_ui_progressStartColor) {
                progressStartColor = typedArray.getColor(attr, 0xFF000000);
            } else if (attr == R.styleable.UIProgressView_ui_progressStopColor) {
                progressStopColor = typedArray.getColor(attr, 0xFF000000);
            } else if (attr == R.styleable.UIProgressView_ui_progress) {
                progress = typedArray.getInt(attr, 0);
                if (progress < minProgress) {
                    progress = (int) minProgress;
                }
            }
        }
        typedArray.recycle();
        paint = new Paint();
        paint.setColor(Color.WHITE);
        paint.setAntiAlias(true);
        paint.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.DST_OUT));
        //第一进度条
        mHintPaint.setColor(hintColor);
        mHintPaint.setAntiAlias(true);
        mHintPaint.setStyle(Paint.Style.FILL);
        //第二进度条
        mPaint.setAntiAlias(true);
        mPaint.setStyle(Paint.Style.FILL);
    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        canvas.drawRect(new Rect(0, 0, getMeasuredWidth(), getHeight()), mHintPaint);
        float progressWidth = (progress / maxProgress) * getMeasuredWidth();
        int y = getHeight() / 2;
        LinearGradient linearGradient = new LinearGradient(0, y, progressWidth, y, getStartColor(), getStopColor(), Shader.TileMode.REPEAT);
        mPaint.setShader(linearGradient);
        canvas.drawRect(new RectF(0, 0, progressWidth, getHeight()), mPaint);
    }

    private int getStartColor() {
        if (progressStartColor == -1 || progressStopColor == -1) {
            return progressColor;
        } else {
            return progressStartColor;
        }
    }

    private int getStopColor() {
        if (progressStartColor == -1 || progressStopColor == -1) {
            return progressColor;
        } else {
            return progressStopColor;
        }
    }

    @Override
    public void draw(Canvas canvas) {
        if (bitmap == null) {
            bitmap = Bitmap.createBitmap(getWidth(), getHeight(), Bitmap.Config.ARGB_8888);
        }
        if (canvas2 == null) {
            canvas2 = new Canvas(bitmap);
        }
        super.draw(canvas2);
        drawLiftUp(canvas2);
        drawLiftDown(canvas2);
        drawRightUp(canvas2);
        drawRightDown(canvas2);
        canvas.drawBitmap(bitmap, 0, 0, mHintPaint);

        canvas.drawBitmap(bitmap, 0, 0, mPaint);
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        if (event.getAction() == MotionEvent.ACTION_MOVE) {
            float moveCount = event.getX() / getWidth();
            int oldProgress = progress;
            progress = (int) (moveCount * maxProgress + minProgress);
            if (progress > maxProgress) {
                progress = (int) maxProgress;
            } else if (progress < minProgress) {
                progress = (int) minProgress;
            }
            if (oldProgress != progress && mListener != null) {
                mListener.change(this, progress);
            }
            invalidate();
        }
        return true;
    }

    public void setOnProgressChangeListener(OnProgressChangeListener listener) {
        mListener = listener;
    }

    public interface OnProgressChangeListener {
        void change(UIProgressView view, int progress);
    }


    private void drawLiftUp(Canvas canvas) {
        int roundWidth = getHeight() / 2;
        Path path = new Path();
        path.moveTo(0, roundWidth);
        path.lineTo(0, 0);
        path.lineTo(roundWidth, 0);
        path.arcTo(new RectF(0, 0, roundWidth * 2, roundWidth * 2), -90, -90);
        path.close();
        canvas.drawPath(path, paint);
    }

    private void drawLiftDown(Canvas canvas) {
        int roundWidth = getHeight() / 2;
        Path path = new Path();
        path.moveTo(0, getHeight() - roundWidth);
        path.lineTo(0, getHeight());
        path.lineTo(roundWidth, getHeight());
        path.arcTo(new RectF(0, getHeight() - roundWidth * 2, roundWidth * 2, getHeight()), 90, 90);
        path.close();
        canvas.drawPath(path, paint);
    }

    private void drawRightDown(Canvas canvas) {
        int roundWidth = getHeight() / 2;
        Path path = new Path();
        path.moveTo(getWidth() - roundWidth, getHeight());
        path.lineTo(getWidth(), getHeight());
        path.lineTo(getWidth(), getHeight() - roundWidth);
        path.arcTo(new RectF(getWidth() - roundWidth * 2, getHeight() - roundWidth * 2, getWidth(), getHeight()), -0, 90);
        path.close();
        canvas.drawPath(path, paint);
    }

    private void drawRightUp(Canvas canvas) {
        int roundWidth = getHeight() / 2;
        Path path = new Path();
        path.moveTo(getWidth(), roundWidth);
        path.lineTo(getWidth(), 0);
        path.lineTo(getWidth() - roundWidth, 0);
        path.arcTo(new RectF(getWidth() - roundWidth * 2, 0, getWidth(), 0 + roundWidth * 2), -90, 90);
        path.close();
        canvas.drawPath(path, paint);
    }
}
