package cn.lsmya.view.appbar;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Color;
import android.util.AttributeSet;
import android.view.MenuItem;
import android.view.View;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.StringRes;
import androidx.appcompat.widget.Toolbar;

import cn.lsmya.view.R;

/**
 * 沉浸式、版本兼容的Toolbar，状态栏透明.
 */
public class UICompatToolbar extends RelativeLayout {
    private Toolbar toolbar;
    private TextView titleView;
    private OnClickListener listener;
    private OnMenuClickListener menuClickListener;

    public UICompatToolbar(Context context) {
        this(context, null);
    }

    public UICompatToolbar(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public UICompatToolbar(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init(context, attrs);
    }

    private void init(Context context, AttributeSet attrs) {
        View view = inflate(getContext(), R.layout.ui_toolbar, this);
        toolbar = view.findViewById(R.id.toolbar_toolbar);
        titleView = view.findViewById(R.id.toolbar_title);
        if (attrs == null) {
            return;
        }
        TypedArray typedArray = context.obtainStyledAttributes(attrs, R.styleable.UICompatToolbar);
        for (int i = 0; i < typedArray.getIndexCount(); i++) {
            int attr = typedArray.getIndex(i);
            if (attr == R.styleable.UICompatToolbar_ui_title) {
                titleView.setText(typedArray.getString(attr));
            } else if (attr == R.styleable.UICompatToolbar_ui_navigationIcon) {
                int icon = typedArray.getResourceId(attr, 0);
                if (0 != icon) {
                    toolbar.setNavigationIcon(icon);
                }
            } else if (attr == R.styleable.UICompatToolbar_ui_menu) {
                int resourceId = typedArray.getResourceId(attr, 0);
                if (resourceId != 0) {
                    toolbar.inflateMenu(resourceId);
                }
            } else if (attr == R.styleable.UICompatToolbar_ui_titleColor) {
                titleView.setTextColor(typedArray.getColor(attr, Color.parseColor("#111111")));
            }
        }
        typedArray.recycle();
        toolbar.setNavigationOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                if (listener != null) {
                    listener.onClick(v);
                }
            }
        });
        toolbar.setOnMenuItemClickListener(new Toolbar.OnMenuItemClickListener() {
            @Override
            public boolean onMenuItemClick(MenuItem menuItem) {
                if (menuClickListener != null) {
                    menuClickListener.onMenuClick(menuItem);
                }
                return true;
            }
        });
    }

    public void setNavigationOnClickListener(OnClickListener listener) {
        this.listener = listener;
    }

    public void setNavigationIcon(int resId) {
        if (this.toolbar != null) {
            this.toolbar.setNavigationIcon(resId);
        }
    }

    public void hideNavigationIcon() {
        if (this.toolbar != null) {
            this.toolbar.setNavigationIcon(null);
        }
    }

    public void setOnMenuClickListener(OnMenuClickListener listener) {
        this.menuClickListener = listener;
    }

    public interface OnMenuClickListener {
        void onMenuClick(MenuItem menuItem);
    }

    public void setTitle(@StringRes int resId) {
        this.setTitle(getContext().getText(resId));
    }

    public void setTitle(CharSequence title) {
        if (titleView != null) {
            titleView.setText(title);
        }
    }

}