package cn.lsmya.view;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Color;
import android.graphics.PorterDuff;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.GradientDrawable;
import android.util.AttributeSet;
import android.view.MotionEvent;

import androidx.annotation.Nullable;
import androidx.appcompat.widget.AppCompatTextView;

/**
 * 支持直接设置圆角、边框的TextView，支持点击效果
 * @author lsm
 */
public class SuperTextView extends AppCompatTextView {

    private static final int OVAL = 0x1;
    private static final int RECTANGLE = 0x2;

    private GradientDrawable mGradientDrawable;

    public SuperTextView(Context context) {
        this(context, null);
    }

    public SuperTextView(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public SuperTextView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init(context, attrs);
    }

    private void init(Context context, @Nullable AttributeSet attrs) {
        TypedArray typedArray = context.obtainStyledAttributes(attrs, R.styleable.kit_superText);
        float corner = typedArray.getDimensionPixelSize(R.styleable.kit_superText_corner, 0);
        float mTopLeftRadius = typedArray.getDimensionPixelSize(R.styleable.kit_superText_topLeft, 0);
        float mTopRightRadius = typedArray.getDimensionPixelSize(R.styleable.kit_superText_topRight, 0);
        float mBottomLeftRadius = typedArray.getDimensionPixelSize(R.styleable.kit_superText_bottomLeft, 0);
        float mBottomRightRadius = typedArray.getDimensionPixelSize(R.styleable.kit_superText_bottomRight, 0);
        int mBorderWidth = typedArray.getDimensionPixelSize(R.styleable.kit_superText_borderWidth, 0);
        int mBorderColor = typedArray.getColor(R.styleable.kit_superText_borderColor, Color.TRANSPARENT);
        int mShape = typedArray.getInt(R.styleable.kit_superText_model, RECTANGLE);

        typedArray.recycle();

        mGradientDrawable = new GradientDrawable();
        mGradientDrawable.setOrientation(GradientDrawable.Orientation.LEFT_RIGHT);
        ColorDrawable background = (ColorDrawable) getBackground();
        if (background != null) {
            int backgroundColor = background.getColor();
            mGradientDrawable.setColor(backgroundColor);
        }
        mGradientDrawable.setStroke(mBorderWidth, mBorderColor);
        if (mTopLeftRadius != 0 || mTopRightRadius != 0 || mBottomLeftRadius != 0 || mBottomRightRadius != 0) {
            mGradientDrawable.setCornerRadii(new float[]{
                    mTopLeftRadius, mTopLeftRadius, mTopRightRadius, mTopRightRadius,
                    mBottomRightRadius, mBottomRightRadius, mBottomLeftRadius, mBottomLeftRadius});
        } else {
            if (corner != 0) {
                mGradientDrawable.setCornerRadius(corner);
            }
        }
        if (mShape == OVAL) {
            mGradientDrawable.setShape(GradientDrawable.OVAL);
        } else {
            mGradientDrawable.setShape(GradientDrawable.RECTANGLE);
        }
        setBackground(mGradientDrawable);
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        if (!isClickable()) {
            return true;
        }
        switch (event.getAction()) {
            case MotionEvent.ACTION_DOWN:
                mGradientDrawable.mutate().setColorFilter(Color.GRAY, PorterDuff.Mode.MULTIPLY);
                setBackground(mGradientDrawable);
                break;
            case MotionEvent.ACTION_UP:
            case MotionEvent.ACTION_CANCEL:
                mGradientDrawable.mutate().clearColorFilter();
                setBackground(mGradientDrawable);
                break;
            default:
                break;
        }
        return super.onTouchEvent(event);
    }
}
