package rx.library.bar;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Color;
import android.graphics.PorterDuff;
import android.graphics.drawable.Drawable;
import android.text.Spannable;
import android.text.SpannableStringBuilder;
import android.text.TextUtils;
import android.text.style.ForegroundColorSpan;
import android.util.AttributeSet;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.MenuItem;
import android.view.View;
import android.widget.FrameLayout;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.appcompat.widget.Toolbar;

import com.google.android.material.appbar.AppBarLayout;

public class BarView extends FrameLayout {
    private AppBarLayout mAppBarLayout;
    private Toolbar mToolbar;
    private LinearLayout mTitleLayout;
    private TextView mTitleView;
    private TextView mSubTitleView;

    public BarView(Context context) {
        this(context, null);
    }

    public BarView(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public BarView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        initView(context, attrs);
    }

    private void initView(Context context, AttributeSet attrs) {
        LayoutInflater.from(context).inflate(R.layout.bar_layout, this);
        findView();
        if (attrs == null) {
            return;
        }
        initToolbar();
        TypedArray typedArray = context.obtainStyledAttributes(attrs, R.styleable.BarView);
        for (int i = 0; i < typedArray.getIndexCount(); i++) {
            int attr = typedArray.getIndex(i);
            if (attr == R.styleable.BarView_title) {
                //标题
                String value = typedArray.getString(attr);
                if (!TextUtils.isEmpty(value)) {
                    mTitleView.setVisibility(VISIBLE);
                    mTitleView.setText(value);
                    getToolbar().setTitle(value);
                }
            } else if (attr == R.styleable.BarView_titleGravity) {
                //标题是否居中
                int value = typedArray.getInt(attr, Gravity.LEFT);
                if (value != Gravity.CENTER) {
                    mTitleLayout.setVisibility(GONE);
                } else {
                    getToolbar().setTitle(null);
                    getToolbar().setSubtitle(null);
                    mTitleLayout.setVisibility(VISIBLE);
                }
            } else if (attr == R.styleable.BarView_subTitle) {
                //副标题
                String value = typedArray.getString(attr);
                if (!TextUtils.isEmpty(value)) {
                    mSubTitleView.setVisibility(VISIBLE);
                    mSubTitleView.setText(value);
                    getToolbar().setSubtitle(value);
                }
            } else if (attr == R.styleable.BarView_titleColor) {
                int color = typedArray.getColor(attr, Color.WHITE);
                getToolbar().setTitleTextColor(color);
                getToolbar().setSubtitleTextColor(color);

                getTitleView().setTextColor(color);
                getSubTitleView().setTextColor(color);

            } else if (attr == R.styleable.BarView_navIcon) {
                int value = typedArray.getResourceId(attr, 0);
                if (value != 0) {
                    getToolbar().setNavigationIcon(value);
                }
            } else if (attr == R.styleable.BarView_showNavIcon) {
                boolean aBoolean = typedArray.getBoolean(attr, true);
                if (!aBoolean) {
                    getToolbar().setNavigationIcon(null);
                }
            } else if (attr == R.styleable.BarView_backgroundColor) {
                int color = typedArray.getColor(attr, Color.WHITE);
                getToolbar().setBackgroundColor(color);
            }
        }

    }

    @Override
    public void setBackground(Drawable background) {
        getToolbar().setBackground(background);
    }

    @Override
    public void setBackgroundColor(int color) {
        getToolbar().setBackgroundColor(color);
    }

    @Override
    public void setBackgroundResource(int resid) {
        getToolbar().setBackgroundResource(resid);
    }

    public Toolbar getToolbar() {
        return mToolbar;
    }

    public TextView getTitleView() {
        return mTitleView;
    }

    public TextView getSubTitleView() {
        return mSubTitleView;
    }

    private void findView() {
        mAppBarLayout = findViewById(R.id.bar_appBarLayout);
        mToolbar = findViewById(R.id.bar_toolbar);
        mTitleLayout = findViewById(R.id.bar_titleLayout);
        mTitleView = findViewById(R.id.bar_title);
        mSubTitleView = findViewById(R.id.bar_subTitle);
    }

    private void initToolbar() {
        getToolbar().setNavigationIcon(R.drawable.kit_back_white_24dp);
    }

    public void setNavigationOnClickListener(final OnNavigationClickListener listener) {
        if (listener != null) {
            getToolbar().setNavigationOnClickListener(new OnClickListener() {
                @Override
                public void onClick(View v) {
                    listener.onNavigationClick();
                }
            });
        }
    }

    public AppBarLayout getAppBarLayout() {
        return mAppBarLayout;
    }

    public void setTitleGravity(int gravity) {
        getTitleView().setGravity(gravity);
    }

    public void addMenuItem(String title, OnMenuClickListener listener) {
        addMenuItem(title, Color.WHITE, listener);
    }

    public void addMenuItem(String title, int color, final OnMenuClickListener listener) {
        if (getToolbar().getMenu() == null) {
            return;
        }
        MenuItem menuItem = getToolbar().getMenu().add(addColor(title, color));
        menuItem.setShowAsAction(MenuItem.SHOW_AS_ACTION_ALWAYS);
        if (listener != null) {
            menuItem.setOnMenuItemClickListener(new MenuItem.OnMenuItemClickListener() {
                @Override
                public boolean onMenuItemClick(MenuItem item) {

                    listener.onClick();
                    return false;
                }
            });
        }
    }

    public void addMenuItem(int iconId, OnMenuClickListener listener) {
        addMenuItem(iconId, Color.WHITE, listener);
    }

    public void addMenuItem(int iconId, int color, final OnMenuClickListener listener) {
        if (getToolbar().getMenu() == null) {
            return;
        }
        MenuItem menuItem = getToolbar().getMenu().add(null);
        menuItem.setIcon(iconId);
        menuItem.getIcon().setColorFilter(color, PorterDuff.Mode.SRC_ATOP);
        menuItem.setShowAsAction(MenuItem.SHOW_AS_ACTION_ALWAYS);
        if (listener != null) {
            menuItem.setOnMenuItemClickListener(new MenuItem.OnMenuItemClickListener() {
                @Override
                public boolean onMenuItemClick(MenuItem item) {
                    listener.onClick();
                    return false;
                }
            });
        }
    }

    private SpannableStringBuilder addColor(CharSequence text, int color) {
        SpannableStringBuilder builder = new SpannableStringBuilder(text);
        if (color != 0) {
            builder.setSpan(
                    new ForegroundColorSpan(color), 0, text.length(),
                    Spannable.SPAN_EXCLUSIVE_EXCLUSIVE
            );
        }
        return builder;
    }


}
