/*
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "{}"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright $date.year Alan Tai
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package android.lib.widget.snackbar;

import java.util.ArrayList;
import java.util.List;

import android.app.Activity;
import android.graphics.Paint;
import android.graphics.Rect;
import android.util.DisplayMetrics;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.widget.FrameLayout;
import android.widget.TextView;

public class Snackbar {
    /**
     * Constant values for defining the placement of the {@link Snackbar} within its container.
     */
    public enum Gravity {
        /** Places the {@link Snackbar} to the top of its container. */
        TOP,
        /** Places the {@link Snackbar} to the bottom of its container. */
        BOTTOM
    }

    /**
     * Shows the text notification for a short period of time. This time could be user-definable. This is the default.
     * @see {#setDuration(int)}
     */
    public static final int LENGTH_SHORT = 2000;

    /**
     * Shows the text notification for a long period of time. This time could be user-definable.
     * @see {#setDuration(int)}
     */
    public static final int LENGTH_LONG  = 3500;

    /**
     * Shows the text notification for an infinite period of time.
     */
    public static final int LENGTH_INFINITE = -1;

    private static final List<Snackbar> SNACKBARS = new ArrayList<>();

    private static boolean isShowing;

    private final Activity  activity;
    private final ViewGroup parent;
    private final View      view;
    private final TextView  message;
    private final TextView  button1;
    private final TextView  button2;

    private Gravity gravity  = Gravity.BOTTOM;
    private int     duration = Snackbar.LENGTH_SHORT;

    private Snackbar(final Activity activity) {
        this.activity = activity;
        this.parent   = (FrameLayout)activity.findViewById(android.R.id.content);
        this.view     = LayoutInflater.from(activity).inflate(R.layout.snackbar, this.parent, false);
        this.message  = (TextView)this.view.findViewById(android.R.id.message);
        this.button1  = (TextView)this.view.findViewById(android.R.id.button1);
        this.button2  = (TextView)this.view.findViewById(android.R.id.button2);
    }

    /**
     * Sets how long to show the text notification for.
     * @param duration The duration to show the text notification for.
     * @see {@link #LENGTH_SHORT}
     * @see {@link #LENGTH_LONG}
     * @return The original {@link Snackbar} object.
     */
    public Snackbar setDuration(final int duration) {
        this.duration = duration;

        return this;
    }

    /**
     * Sets the placement of the {@link Snackbar} within its container.
     * @param gravity The placement, either {@link Gravity#TOP} or {@link Gravity#BOTTOM}.
     * @see {@link Gravity#TOP}
     * @see {@link Gravity#BOTTOM}
     * @return The original {@link Snackbar} object.
     */
    public Snackbar setGravity(final Gravity gravity) {
        this.gravity = gravity;

        return this;
    }

    /**
     * Sets the string value of the {@link Snackbar}.
     * @param resId The string resource ID to set.
     * @return The original {@link Snackbar} object.
     */
    public Snackbar setText(final int resId) {
        this.message.setText(resId);

        return this;
    }

    /**
     * Sets the string value of the {@link Snackbar}.
     * @param s The string value to set.
     * @return The original {@link Snackbar} object.
     */
    public Snackbar setText(final CharSequence s) {
        this.message.setText(s);

        return this;
    }

    /**
     * Sets the text color of the string to be displayed.
     * @param color The text color of the string to be displayed.
     * @return The original {@link Snackbar} object.
     */
    public Snackbar setTextColor(final int color) {
        this.message.setTextColor(color);

        return this;
    }

    /**
     * Sets the action button text and text color, and the botton's click event listener.
     * @param resId The action button text resource ID.
     * @param color The action button text color.
     * @param listener The action button click event listener.
     * @return The original {@link Snackbar} object.
     */
    public Snackbar setAction(final int resId, final int color, final View.OnClickListener listener) {
        this.button1.setText(resId);
        this.button1.setTextColor(color);
        this.button1.setOnClickListener(listener);
        this.button1.setVisibility(View.VISIBLE);
        this.button2.setVisibility(View.GONE);

        return this;
    }

    /**
     * Sets the action button text and text color, and the botton's click event listener.
     * @param action The action button text.
     * @param color The action button text color.
     * @param listener The action button click event listener.
     * @return The original {@link Snackbar} object.
     */
    public Snackbar setAction(final CharSequence action, final int color, final View.OnClickListener listener) {
        this.button1.setText(action);
        this.button1.setTextColor(color);
        this.button1.setOnClickListener(listener);
        this.button1.setVisibility(View.VISIBLE);
        this.button2.setVisibility(View.GONE);

        return this;
    }

    /**
     * Sets the text, text color, and the click event listeners for the two action buttons.
     * @param resId1 The text resource ID for the first action button.
     * @param color1 The text color for the first action button.
     * @param listener1 The click event listener for the first action button.
     * @param resId2 The text resource ID for the second action button.
     * @param color2 The text color for the second action button.
     * @param listener2 The click event listener for the second action button.
     * @return The original {@link Snackbar} object.
     */
    public Snackbar setActions(final int resId1, final int color1, final View.OnClickListener listener1, final int resId2, final int color2, final View.OnClickListener listener2) {
        this.button1.setText(resId1);
        this.button2.setText(resId2);
        this.button1.setTextColor(color1);
        this.button2.setTextColor(color2);
        this.button1.setOnClickListener(listener1);
        this.button2.setOnClickListener(listener2);
        this.button1.setVisibility(View.VISIBLE);
        this.button2.setVisibility(View.VISIBLE);

        return this;
    }

    /**
     * Shows the {@link Snackbar}.
     */
    public synchronized void show() {
        if (Snackbar.isShowing) {
            Snackbar.SNACKBARS.add(this);
        } else {
            Snackbar.isShowing = true;

            final DisplayMetrics metrics = new DisplayMetrics();
            this.activity.getWindow().getWindowManager().getDefaultDisplay().getMetrics(metrics);

            final Rect  messageRect  = new Rect();
            final Paint messagePaint = new Paint();

            messagePaint.setTextSize(this.message.getTextSize());
            messagePaint.setTypeface(this.message.getTypeface());
            messagePaint.getTextBounds(this.message.getText().toString(), 0, this.message.getText().length(), messageRect);

            final int margin = this.activity.getResources().getDimensionPixelSize(R.dimen.snackbar_margin);

            int button1Width = 0;
            int button2Width = 0;

            if (this.button1.getVisibility() == View.VISIBLE) {
                final Rect  buttonRect  = new Rect();
                final Paint buttonPaint = new Paint();

                buttonPaint.setTextSize(this.message.getTextSize());
                buttonPaint.setTypeface(this.message.getTypeface());
                buttonPaint.getTextBounds(this.message.getText().toString(), 0, this.message.getText().length(), buttonRect);

                button1Width = buttonRect.width() + margin;
            }

            if (this.button2.getVisibility() == View.VISIBLE) {
                final Rect  buttonRect  = new Rect();
                final Paint buttonPaint = new Paint();

                buttonPaint.setTextSize(this.message.getTextSize());
                buttonPaint.setTypeface(this.message.getTypeface());
                buttonPaint.getTextBounds(this.message.getText().toString(), 0, this.message.getText().length(), buttonRect);

                button2Width = buttonRect.width() + margin;
            }

            // Center the message if there is no action buttons
            if (this.button1.getVisibility() != View.VISIBLE && this.button2.getVisibility() != View.VISIBLE) {
                this.message.setGravity(android.view.Gravity.CENTER);
            }

            final int height;

            if (metrics.widthPixels >= messageRect.width() - margin - button1Width - button2Width) {
                height = this.activity.getResources().getDimensionPixelSize(R.dimen.snackbar_height_1);
            } else {
                height = this.activity.getResources().getDimensionPixelSize(R.dimen.snackbar_height_2);
            }

            final FrameLayout.LayoutParams params = new FrameLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, height);
            params.gravity = this.gravity == Gravity.BOTTOM ? android.view.Gravity.BOTTOM : android.view.Gravity.TOP;

            this.parent.addView(this.view, params);

            final Animation animation = AnimationUtils.loadAnimation(this.activity, this.gravity == Gravity.BOTTOM ? R.anim.snackbar_in_from_bottom : R.anim.snackbar_in_from_top);

            animation.setAnimationListener(new Animation.AnimationListener() {
                @Override
                public void onAnimationStart(final Animation animation) {
                }

                @Override
                public void onAnimationEnd(final Animation animation) {
                    if (Snackbar.this.duration > 0) {
                        Snackbar.this.view.postDelayed(new HideRunnable(), Snackbar.this.duration);
                    }
                }

                @Override
                public void onAnimationRepeat(final Animation animation) {
                }
            });

            this.view.startAnimation(animation);
        }
    }

    /**
     * Hides the {@link Snackbar} if it is currently showing; otherwise, does nothing.
     */
    public void hide() {
        if (Snackbar.SNACKBARS.contains(this)) {
            Snackbar.SNACKBARS.remove(this);
        }

        if (Snackbar.isShowing) {
            new HideRunnable().run();
        }
    }

    /**
     * Creates a {@link Snackbar} with the given {@link Activity} as its container.
     * @param activity The container.
     */
    public static Snackbar with(final Activity activity) {
        return new Snackbar(activity);
    }

    private final class HideRunnable implements Runnable {
        @Override
        public void run() {
            final Animation animation = AnimationUtils.loadAnimation(Snackbar.this.activity, Snackbar.this.gravity == Gravity.BOTTOM ? R.anim.snackbar_out_from_bottom : R.anim.snackbar_out_from_top);

            animation.setAnimationListener(new Animation.AnimationListener() {
                @Override
                public void onAnimationStart(final Animation animation) {
                }

                @Override
                public void onAnimationEnd(final Animation animation) {
                    Snackbar.this.parent.removeView(Snackbar.this.view);

                    Snackbar.isShowing = false;

                    if (!Snackbar.SNACKBARS.isEmpty()) {
                        Snackbar.SNACKBARS.remove(0).show();
                    }
                }

                @Override
                public void onAnimationRepeat(final Animation animation) {
                }
            });

            Snackbar.this.view.startAnimation(animation);
        }
    }
}
