/*
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "{}"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2016 Alan Tai
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package android.lib.recaptcha;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.net.http.AndroidHttpClient;
import android.os.AsyncTask;
import android.os.Handler;
import android.os.Message;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.util.Log;
import android.widget.ImageView;

import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.BasicResponseHandler;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.params.HttpParams;
import org.apache.http.protocol.HTTP;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * {@code ReCaptcha} extends {@link android.widget.ImageView} to let you embed a <a href="http://captcha.net/">CAPTCHA</a>
 * in your applications in order to protect them against spam and other types of automated abuse.
 *
 * @see <a href="https://developers.google.com/recaptcha/">reCAPTCHA</a>
 */
public class ReCaptcha extends ImageView {
    /**
     * Listener that is called when an attempt to show a <a href="http://captcha.net/">CAPTCHA</a>
     * is completed.
     */
    public interface OnShowChallengeListener {
        /**
         * Called when an attempt to show a <a href="http://captcha.net/">CAPTCHA</a> is completed.
         *
         * @param shown {@code true} if a <a href="http://captcha.net/">CAPTCHA</a> is shown;
         *              otherwise, {@code false}.
         */
        void onChallengeShown(boolean shown);
    }

    /**
     * Listener that is called when an answer entered by the user to solve the <a href="http://captcha.net/">CAPTCHA</a>
     * displayed is verified.
     */
    public interface OnVerifyAnswerListener {
        /**
         * Called when an answer entered by the user to solve the <a href="http://captcha.net/">CAPTCHA</a>
         * displayed is verified.
         *
         * @param success {@code true} if the <a href="http://captcha.net/">CAPTCHA</a> is solved successfully;
         *                otherwise, {@code false}.
         */
        void onAnswerVerified(boolean success);
    }

    private static final String TAG = "ReCaptcha";

    private static final String VERIFICATION_URL           = "http://www.google.com/recaptcha/api/verify";
    private static final String CHALLENGE_URL              = "http://www.google.com/recaptcha/api/challenge?k=%s";
    private static final String RECAPTCHA_OBJECT_TOKEN_URL = "http://www.google.com/recaptcha/api/reload?c=%s&k=%s&type=%s";
    private static final String IMAGE_URL                  = "http://www.google.com/recaptcha/api/image?c=%s";

    private String challenge;
    private String imageToken;

    public ReCaptcha(final Context context) {
        super(context);
    }

    public ReCaptcha(final Context context, final AttributeSet attrs) {
        super(context, attrs);
    }

    public ReCaptcha(final Context context, final AttributeSet attrs, final int defStyle) {
        super(context, attrs, defStyle);
    }

    /**
     * Returns a new instance of {@link HttpClient} for downloading ReCaptcha images.
     * <p>Subclasses may override this method and return a customized {@link HttpClient},
     * such as an {@link AndroidHttpClient} with custom {@link HttpParams}.</p>
     * <p>The default behavior returns a {@link DefaultHttpClient}</p>
     *
     * @return a {@link HttpClient} for downloading ReCaptcha images.
     */
    protected HttpClient createHttpClient() {
        return new DefaultHttpClient();
    }

    /**
     * Returns the challenge previously returned from the server.
     * <p>The challenge previously returned from the server will be available after {@link #showChallenge(String)} returns or after {@link ReCaptcha.OnShowChallengeListener#onChallengeShown(boolean)} is called.</p>
     * @return The challenge previously returned from the server.
     */
    public final String getChallenge() {
        return this.challenge;
    }

    /**
     * Returns the image token previously returned from the server.
     * <p>The image token previously returned from the server will be available after {@link #showChallenge(String)} returns or after {@link ReCaptcha.OnShowChallengeListener#onChallengeShown(boolean)} is called.</p>
     * @return The image token previously returned from the server.
     */
    public final String getImageToken() {
        return this.imageToken;
    }

    /**
     * Downloads and shows a <a href="http://captcha.net/">CAPTCHA</a> image.
     * <p>You can obtain a public key by <a href="http://www.google.com/recaptcha/whyrecaptcha">signing up for API keys</a>
     * for your application.</p>
     * <p>This method executes synchronously and should not be invoked from the UI thread.
     * For asynchronous invocation, you may use {@link android.os.AsyncTask} or call
     * {@link #showChallengeAsync(String, ReCaptcha.OnShowChallengeListener)} instead.</p>
     *
     * @param publicKey The public key that is unique to your domain and sub-domains (unless it is a global key).
     * @return {@code true} if a <a href="http://captcha.net/">CAPTCHA</a> {@link android.graphics.Bitmap}
     * is successfully downloaded and shown; otherwise, {@code false}.
     * @throws android.lib.recaptcha.ReCaptchaException if the downloaded <a href="http://captcha.net/">CAPTCHA</a> content is malformed.
     * @throws java.io.IOException                      in case of a protocol or network connection problem.
     * @see #showChallengeAsync(String, ReCaptcha.OnShowChallengeListener)
     */
    public final boolean showChallenge(final String publicKey) throws ReCaptchaException, IOException {
        if (TextUtils.isEmpty(publicKey)) {
            throw new IllegalArgumentException("publicKey cannot be null or empty");
        }

        this.setImageDrawable(null);

        this.challenge  = null;
        this.imageToken = null;

        final Bitmap bitmap = this.downloadImage(publicKey);

        this.setImageBitmap(bitmap);

        return bitmap != null;
    }

    /**
     * Downloads and shows a <a href="http://captcha.net/">CAPTCHA</a> image asynchronously.
     * <p>This method executes asynchronously and can be invoked from the UI thread.
     * For synchronous invocation, use {@link #showChallenge(String)}.</p>
     *
     * @param publicKey The public key that is unique to your domain and sub-domains (unless it is global key).
     * @param listener  The callback to call when an attempt to show a <a href="http://captcha.net/">CAPTCHA</a> is completed.
     * @see #showChallenge(String)
     */
    public final void showChallengeAsync(final String publicKey, final ReCaptcha.OnShowChallengeListener listener) {
        if (TextUtils.isEmpty(publicKey)) {
            throw new IllegalArgumentException("publicKey cannot be null or empty");
        }

        this.setImageDrawable(null);

        this.challenge  = null;
        this.imageToken = null;

        final Handler handler = new Handler() {
            @Override
            public void handleMessage(final Message message) {
                final Bitmap bitmap = (Bitmap) message.obj;
                final Bitmap scaled = Bitmap.createScaledBitmap(bitmap, bitmap.getWidth() * 2, bitmap.getHeight() * 2, true);

                bitmap.recycle();

                ReCaptcha.this.setImageBitmap(scaled);

                if (listener != null) {
                    listener.onChallengeShown(message.obj != null);
                }
            }
        };

        new AsyncTask<String, Void, Bitmap>() {
            @Override
            protected Bitmap doInBackground(final String... publicKeys) {
                try {
                    return ReCaptcha.this.downloadImage(publicKeys[0]);
                } catch (final ReCaptchaException e) {
                    Log.e(ReCaptcha.TAG, "The downloaded CAPTCHA content is malformed", e);
                } catch (final IOException e) {
                    Log.e(ReCaptcha.TAG, "A protocol or network connection problem has occurred", e);
                }

                return null;
            }

            @Override
            protected void onPostExecute(final Bitmap bitmap) {
                handler.sendMessage(handler.obtainMessage(0, bitmap));
            }
        }.execute(publicKey);
    }

    /**
     * Checks whether the answer entered by the user is correct after your application is successfully
     * displaying <a href="https://developers.google.com/recaptcha/">reCAPTCHA</a>.
     * <p>This method executes synchronously and should not be invoked from the UI thread.
     * For asynchronous invocation, you may use {@link android.os.AsyncTask} or call
     * {@link #verifyAnswerAsync(String, String, ReCaptcha.OnVerifyAnswerListener)} instead.</p>
     *
     * @param privateKey The private key that is unique to your domain and sub-domains (unless it is a global key).
     * @param answer     The string the user entered to solve the <a href="http://captcha.net/">CAPTCHA</a> displayed.
     * @return {@code true} if the <a href="https://developers.google.com/recaptcha/">reCAPTCHA</a> was successful;
     * otherwise {@code false}.
     * @throws java.io.IOException in case of a protocol or network connection problem.
     * @see #verifyAnswerAsync(String, String, ReCaptcha.OnVerifyAnswerListener)
     */
    public final boolean verifyAnswer(final String privateKey, final String answer) throws IOException {
        if (TextUtils.isEmpty(privateKey)) {
            throw new IllegalArgumentException("privateKey cannot be null or empty");
        }

        if (TextUtils.isEmpty(answer)) {
            throw new IllegalArgumentException("answer cannot be null or empty");
        }

        return this.submitAnswer(privateKey, answer);
    }

    /**
     * Checks asynchronously whether the answer entered by the user is correct after your application
     * is successfully displaying <a href="https://developers.google.com/recaptcha/">reCAPTCHA</a>.
     *
     * @param privateKey The private key that is unique to your domain and sub-domains (unless it is a global key).
     * @param answer     The string the user entered to solve the <a href="http://captcha.net/">CAPTCHA</a> displayed.
     * @param listener   The callback to call when an answer entered by the user is verified.
     * @see #verifyAnswer(String, String)
     */
    public final void verifyAnswerAsync(final String privateKey, final String answer, final ReCaptcha.OnVerifyAnswerListener listener) {
        if (TextUtils.isEmpty(privateKey)) {
            throw new IllegalArgumentException("privateKey cannot be null or empty");
        }

        if (TextUtils.isEmpty(answer)) {
            throw new IllegalArgumentException("answer cannot be null or empty");
        }

        final Handler handler = new Handler() {
            @Override
            public void handleMessage(final Message message) {
                if (listener != null) {
                    listener.onAnswerVerified((Boolean) message.obj);
                }
            }
        };

        new AsyncTask<String, Void, Boolean>() {
            @Override
            protected Boolean doInBackground(final String... params) {
                try {
                    return ReCaptcha.this.submitAnswer(params[0], params[1]);
                } catch (final IOException e) {
                    Log.e(ReCaptcha.TAG, "A protocol or network connection problem has occurred", e);
                }

                return Boolean.FALSE;
            }

            @Override
            protected void onPostExecute(final Boolean result) {
                handler.sendMessage(handler.obtainMessage(0, result));
            }
        }.execute(privateKey, answer);
    }

    private Bitmap downloadImage(final String publicKey) throws ReCaptchaException, IOException {
        final HttpClient httpClient = this.createHttpClient();

        try {
            final String challenge = this.getChallenge(publicKey);
            Log.d(ReCaptcha.TAG, "challenge = " + challenge);
            if (challenge == null) {
                throw new ReCaptchaException("ReCaptcha challenge not found");
            }

            final String imageToken = this.getImageToken(challenge, publicKey);
            Log.d(ReCaptcha.TAG, "imageToken = " + imageToken);
            if (imageToken == null) {
                throw new ReCaptchaException("Image token not found");
            }

            this.imageToken = imageToken;

            final String       imageUrl = String.format(ReCaptcha.IMAGE_URL, imageToken);
            final HttpResponse response = httpClient.execute(new HttpGet(imageUrl));

            try {
                final Bitmap bitmap = BitmapFactory.decodeStream(response.getEntity().getContent());

                if (bitmap == null) {
                    throw new ReCaptchaException("Invalid CAPTCHA image");
                }

                return bitmap;
            } finally {
                if (response.getEntity() != null) {
                    response.getEntity().consumeContent();
                }
            }
        } catch (final JSONException e) {
            throw new ReCaptchaException("Unable to parse challenge response", e);
        } finally {
            httpClient.getConnectionManager().shutdown();
        }
    }

    private String getImageToken(final String challenge, final String publicKey) throws IOException {
        final HttpClient httpClient = this.createHttpClient();

        try {
            final String imageTokenResponse = httpClient.execute(new HttpGet(String.format(ReCaptcha.RECAPTCHA_OBJECT_TOKEN_URL, challenge, publicKey, "image")), new BasicResponseHandler());
            Log.d(ReCaptcha.TAG, "imageTokenResponse = " + imageTokenResponse);

            return ReCaptcha.substringBetween(imageTokenResponse, "('", "',");
        } finally {
            httpClient.getConnectionManager().shutdown();
        }
    }

    private String getChallenge(final String publicKey) throws IOException, JSONException {
        final HttpClient httpClient = this.createHttpClient();

        try {
            final String challengeResponse = httpClient.execute(new HttpGet(String.format(ReCaptcha.CHALLENGE_URL, publicKey)), new BasicResponseHandler());
            Log.d(ReCaptcha.TAG, "challengeResponse = " + challengeResponse);

            this.challenge = new JSONObject(ReCaptcha.substringBetween(challengeResponse, "RecaptchaState = ", "}") + "}").getString("challenge");;

            return this.challenge;
        } finally {
            httpClient.getConnectionManager().shutdown();
        }
    }

    private boolean submitAnswer(final String privateKey, final String answer) throws IOException {
        final HttpClient          httpClient = this.createHttpClient();
        final HttpPost            request    = new HttpPost(ReCaptcha.VERIFICATION_URL);
        final List<NameValuePair> params     = new ArrayList<>();

        params.add(new BasicNameValuePair("privatekey", privateKey));
        params.add(new BasicNameValuePair("remoteip", "127.0.0.1"));
        params.add(new BasicNameValuePair("challenge", this.imageToken));
        params.add(new BasicNameValuePair("response", answer));

        try {
            request.setEntity(new UrlEncodedFormEntity(params, HTTP.UTF_8));

            return httpClient.execute(request, new BasicResponseHandler()).startsWith("true");
        } catch (final UnsupportedEncodingException e) {
            Log.e(ReCaptcha.TAG, "UTF-8 encoding is not supported on this platform", e);
        } finally {
            httpClient.getConnectionManager().shutdown();
        }

        return false;
    }

    private static String substringBetween(final String str, final String open, final String close) {
        if (str == null || open == null || close == null) {
            return null;
        }

        final int start = str.indexOf(open);

        if (start != -1) {
            final int end = str.indexOf(close, start + open.length());

            if (end != -1) {
                return str.substring(start + open.length(), end);
            }
        }

        return null;
    }
}
