/*
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "{}"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright $date.year Alan Tai
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package android.lib.database;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import android.content.Context;
import android.content.pm.PackageManager.NameNotFoundException;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.text.TextUtils;
import android.util.Log;

/**
 * Handles database table creation.
 */
public class DatabaseOpenHelper extends SQLiteOpenHelper {
    private static final String CREATE_TABLE        = "CREATE TABLE IF NOT EXISTS %s (%s)";                     //$NON-NLS-1$
    private static final String CREATE_INDEX        = "CREATE INDEX IF NOT EXISTS idx_%s_%s ON %s (%s)";        //$NON-NLS-1$
    private static final String CREATE_INDEX_UNIQUE = "CREATE UNIQUE INDEX IF NOT EXISTS idx_%s_%s ON %s (%s)"; //$NON-NLS-1$
    private static final String PRIMARY_KEY         = "PRIMARY KEY";                                            //$NON-NLS-1$
    private static final String AUTO_INCREMENT      = "AUTOINCREMENT";                                          //$NON-NLS-1$
    private static final String NOT_NULL            = "NOT NULL";                                               //$NON-NLS-1$
    private static final String UNIQUE              = "UNIQUE";                                                 //$NON-NLS-1$
    private static final String COLUMN_SEPARATOR    = ", ";                                                     //$NON-NLS-1$
    private static final String SPACE               = " ";                                                      //$NON-NLS-1$
    private static final String TYPE_INTEGER        = "INTEGER";                                                //$NON-NLS-1$
    private static final String TYPE_REAL           = "REAL";                                                   //$NON-NLS-1$
    private static final String TYPE_TEXT           = "TEXT";                                                   //$NON-NLS-1$
    private static final String TYPE_BLOB           = "BLOB";                                                   //$NON-NLS-1$

    private static final Comparator<CompositeIndex> COMPOSITE_INDEX_COMPARATOR = new Comparator<CompositeIndex>() {
        @Override
        public int compare(final CompositeIndex lhs, final CompositeIndex rhs) {
            return lhs.order() - rhs.order();
        }
    };

    private static final Comparator<UniqueCompositeIndex> UNIQUE_COMPOSITE_INDEX_COMPARATOR = new Comparator<UniqueCompositeIndex>() {
        @Override
        public int compare(final UniqueCompositeIndex lhs, final UniqueCompositeIndex rhs) {
            return lhs.order() - rhs.order();
        }
    };

    /**
     * The tables of the database to create in {@link #onCreate(SQLiteDatabase)}.
     */
    protected final Class<?>[] tables;

    /**
     * Creates a new instance of {@link DatabaseOpenHelper} by invoking
     * its super-class {@link SQLiteOpenHelper}, and optionally stores
     * the given <code>tables</code> for database table creation when {@link #onCreate(SQLiteDatabase)} is called.
     * @param context the context to use to open or create the database
     * @param name the file name of the database file, or null for an in-memory database
     * @param version the version number of the database (starting at 1); if the database is older,
     * {@link #onUpgrade(SQLiteDatabase, int, int)} will be used to upgrade the database;
     * if the database is newer, {@link #onDowngrade(SQLiteDatabase, int, int)} will be used to downgrade the database
     * @param tables the tables of the database to create in {@link #onCreate(SQLiteDatabase)}.
     * For subsequent call, {@code tables} can be {@code null}.
     */
    protected DatabaseOpenHelper(final Context context, final String name, final int version, final Class<?>... tables) {
        super(context, name, null, version);

        this.tables = tables;
    }

    /**
     * Creates a new instance of {@link DatabaseOpenHelper} with the given {@code name} and {@code version}.
     * @param context the context to use to open or create the database
     * @param name the file name of the database file, or null for an in-memory database
     * @param version the version number of the database (starting at 1); if the database is older,
     * {@link #onUpgrade(SQLiteDatabase, int, int)} will be used to upgrade the database;
     * if the database is newer, {@link #onDowngrade(SQLiteDatabase, int, int)} will be used to downgrade the database
     * @param tables the tables of the database to create in {@link #onCreate(SQLiteDatabase)}.
     * For subsequent call, {@code tables} can be {@code null}.
     */
    public static DatabaseOpenHelper newInstance(final Context context, final String name, final int version, final Class<?>... tables) {
        return new DatabaseOpenHelper(context, name, version, tables);
    }

    /**
     * Creates a new instance of {@link DatabaseOpenHelper} with the given {@code version}.
     * <p>The application package name will be used as the name of the database.</p>
     * @param context the context to use to open or create the database
     * @param version the version number of the database (starting at 1); if the database is older,
     * {@link #onUpgrade(SQLiteDatabase, int, int)} will be used to upgrade the database;
     * if the database is newer, {@link #onDowngrade(SQLiteDatabase, int, int)} will be used to downgrade the database
     * @param tables the tables of the database to create in {@link #onCreate(SQLiteDatabase)}.
     * For subsequent call, {@code tables} can be {@code null}.
     */
    public static DatabaseOpenHelper newInstance(final Context context, final int version, final Class<?>... tables) {
        return new DatabaseOpenHelper(context, context.getPackageName(), version, tables);
    }

    /**
     * Creates a new instance of {@link DatabaseOpenHelper} with the given {@code name}.
     * <p>The application version code will be used as the database version.</p>
     * @param context the context to use to open or create the database
     * @param name the file name of the database file, or null for an in-memory database
     * @param tables the tables of the database to create in {@link #onCreate(SQLiteDatabase)}.
     * For subsequent call, {@code tables} can be {@code null}.
     */
    public static DatabaseOpenHelper newInstance(final Context context, final String name, final Class<?>... tables) {
        try {
            return new DatabaseOpenHelper(
                            context,
                            name,
                            context.getPackageManager().getPackageInfo(context.getPackageName(), 0).versionCode,
                            tables);
        } catch (final NameNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Creates a new instance of {@link DatabaseOpenHelper}.
     * <p>The application package name and version code will be used as the name and version of the database respectively.</p>
     * @param context the context to use to open or create the database
     * @param tables the tables of the database to create in {@link #onCreate(SQLiteDatabase)}.
     * For subsequent call, {@code tables} can be {@code null}.
     */
    public static DatabaseOpenHelper newInstance(final Context context, final Class<?>... tables) {
        try {
            return new DatabaseOpenHelper(
                            context,
                            context.getPackageName(),
                            context.getPackageManager().getPackageInfo(context.getPackageName(), 0).versionCode,
                            tables);
        } catch (final NameNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void onCreate(final SQLiteDatabase database) {
        if (this.tables != null) {
            database.beginTransaction();

            try {
                for (final Class<?> table : this.tables) {
                    DatabaseOpenHelper.createTable(database, table);
                }

                database.setTransactionSuccessful();
            } catch (final UnsupportedTypeException e) {
                Log.e(this.getClass().getName(), e.getMessage(), e);
            } finally {
                database.endTransaction();
            }
        }
    }

    @Override
    public void onUpgrade(final SQLiteDatabase database, final int oldVersion, final int newVersion) {
    }

    private static void createTable(final SQLiteDatabase database, final Class<?> clazz) throws UnsupportedTypeException {
        final Table table = clazz.getAnnotation(Table.class);

        if (table != null) {
            final String                                            tableName              = TextUtils.isEmpty(table.value()) ? clazz.getSimpleName() : table.value();
            final StringBuilder                                     columnsBuilder         = new StringBuilder();
            final List<String>                                      createIndexStatements  = new ArrayList<>();
            final Set<Field>                                        fields                 = new HashSet<>();
            final Map<Field, String>                                fieldNames             = new HashMap<>();
            final Map<String, TreeMap<CompositeIndex, Field>>       compositeIndexes       = new TreeMap<>();
            final Map<String, TreeMap<UniqueCompositeIndex, Field>> uniqueCompositeIndexes = new TreeMap<>();

            for (final Field field : clazz.getFields()) {
                fields.add(field);
            }

            for (final Field field : clazz.getDeclaredFields()) {
                fields.add(field);
            }

            for (final Field field : fields) {
                field.setAccessible(true);

                final Column column = field.getAnnotation(Column.class);

                if (column != null) {
                    if (columnsBuilder.length() > 0) {
                        columnsBuilder.append(DatabaseOpenHelper.COLUMN_SEPARATOR);
                    }

                    columnsBuilder.append(DatabaseOpenHelper.getColumnDefinition(field));

                    final String fieldName = TextUtils.isEmpty(column.value()) ? field.getName() : column.value();
                    final Index  index     = field.getAnnotation(Index.class);

                    fieldNames.put(field, fieldName);

                    if (index != null) {
                        createIndexStatements.add(String.format(index.unique() ? DatabaseOpenHelper.CREATE_INDEX_UNIQUE : DatabaseOpenHelper.CREATE_INDEX, tableName, fieldName, tableName, fieldName));
                    }

                    // Searches for composite index annotation
                    final CompositeIndex compositeIndex = field.getAnnotation(CompositeIndex.class);

                    if (compositeIndex != null) {
                        final String[] indexNames = compositeIndex.value() == null ? new String[] { fieldName } : compositeIndex.value();

                        for (final String indexName : indexNames) {
                            final TreeMap<CompositeIndex, Field> indexes = compositeIndexes.containsKey(indexName) ? compositeIndexes.get(indexName) : new TreeMap<CompositeIndex, Field>(DatabaseOpenHelper.COMPOSITE_INDEX_COMPARATOR);

                            indexes.put(compositeIndex, field);
                            compositeIndexes.put(indexName, indexes);
                        }
                    }

                    // Searches for unique composite index annotation
                    final UniqueCompositeIndex uniqueCompositeIndex = field.getAnnotation(UniqueCompositeIndex.class);

                    if (uniqueCompositeIndex != null) {
                        final String[] indexNames = uniqueCompositeIndex.value() == null ? new String[] { fieldName } : uniqueCompositeIndex.value();

                        for (final String indexName : indexNames) {
                            final TreeMap<UniqueCompositeIndex, Field> indexes = uniqueCompositeIndexes.containsKey(indexName) ? uniqueCompositeIndexes.get(indexName) : new TreeMap<UniqueCompositeIndex, Field>(DatabaseOpenHelper.UNIQUE_COMPOSITE_INDEX_COMPARATOR);

                            indexes.put(uniqueCompositeIndex, field);
                            uniqueCompositeIndexes.put(indexName, indexes);
                        }
                    }
                }
            }

            // Creates table
            database.execSQL(String.format(DatabaseOpenHelper.CREATE_TABLE, tableName, columnsBuilder.toString()));

            // Creates indexes
            for (final String createIndexStatement : createIndexStatements) {
                database.execSQL(createIndexStatement);
            }

            // Creates composite indexes
            if (!compositeIndexes.isEmpty()) {
                for (final Map.Entry<String, TreeMap<CompositeIndex, Field>> entry : compositeIndexes.entrySet()) {
                    final StringBuilder indexBuilder = new StringBuilder();

                    for (final Map.Entry<CompositeIndex, Field> index : entry.getValue().entrySet()) {
                        if (indexBuilder.length() > 0) {
                            indexBuilder.append(DatabaseOpenHelper.COLUMN_SEPARATOR);
                        }

                        indexBuilder.append(fieldNames.get(index.getValue()));
                    }

                    database.execSQL(String.format(DatabaseOpenHelper.CREATE_INDEX, tableName, entry.getKey(), tableName, indexBuilder.toString()));
                }
            }

            // Creates unique composite indexes
            if (!uniqueCompositeIndexes.isEmpty()) {
                for (final Map.Entry<String, TreeMap<UniqueCompositeIndex, Field>> entry : uniqueCompositeIndexes.entrySet()) {
                    final StringBuilder indexBuilder = new StringBuilder();

                    for (final Map.Entry<UniqueCompositeIndex, Field> index : entry.getValue().entrySet()) {
                        if (indexBuilder.length() > 0) {
                            indexBuilder.append(DatabaseOpenHelper.COLUMN_SEPARATOR);
                        }

                        indexBuilder.append(fieldNames.get(index.getValue()));
                    }

                    database.execSQL(String.format(DatabaseOpenHelper.CREATE_INDEX_UNIQUE, tableName, entry.getKey(), tableName, indexBuilder.toString()));
                }
            }
        }
    }

    private static String getColumnDefinition(final Field field) throws UnsupportedTypeException {
        final Column        column  = field.getAnnotation(Column.class);
        final StringBuilder builder = new StringBuilder();

        builder.append(TextUtils.isEmpty(column.value()) ? field.getName() : column.value());
        builder.append(DatabaseOpenHelper.SPACE);
        builder.append(DatabaseOpenHelper.getColumnType(field));

        if (column.primaryKey()) {
            builder.append(DatabaseOpenHelper.SPACE);
            builder.append(DatabaseOpenHelper.PRIMARY_KEY);

            if (column.autoIncrement()) {
                builder.append(DatabaseOpenHelper.SPACE);
                builder.append(DatabaseOpenHelper.AUTO_INCREMENT);
            }
        } else {
            if (!column.nullable()) {
                builder.append(DatabaseOpenHelper.SPACE);
                builder.append(DatabaseOpenHelper.NOT_NULL);
            }
        }

        final Index index = field.getAnnotation(Index.class);

        if (index != null && index.unique()) {
            builder.append(DatabaseOpenHelper.SPACE);
            builder.append(DatabaseOpenHelper.UNIQUE);
        }

        return builder.toString();
    }

    private static String getColumnType(final Field field) throws UnsupportedTypeException {
        final UseConverter converter = field.getAnnotation(UseConverter.class);
        final Class<?>     type      = converter == null ? field.getType() : converter.type();

        if (type.equals(boolean.class) || type.equals(Boolean.class)) {
            return DatabaseOpenHelper.TYPE_INTEGER;
        } else if (type.equals(byte.class) || type.equals(Byte.class)) {
            return DatabaseOpenHelper.TYPE_INTEGER;
        } else if (type.equals(byte[].class) || type.equals(Byte[].class)) {
            return DatabaseOpenHelper.TYPE_BLOB;
        } else if (type.equals(double.class) || type.equals(Double.class)) {
            return DatabaseOpenHelper.TYPE_REAL;
        } else if (type.equals(float.class) || type.equals(Float.class)) {
            return DatabaseOpenHelper.TYPE_REAL;
        } else if (type.equals(int.class) || type.equals(Integer.class)) {
            return DatabaseOpenHelper.TYPE_INTEGER;
        } else if (type.equals(long.class) || type.equals(Long.class)) {
            return DatabaseOpenHelper.TYPE_INTEGER;
        } else if (type.equals(short.class) || type.equals(Short.class)) {
            return DatabaseOpenHelper.TYPE_INTEGER;
        } else if (type.equals(String.class)) {
            return DatabaseOpenHelper.TYPE_TEXT;
        } else {
            throw new UnsupportedTypeException(String.format("%s is not of SQLite-compatible type", field.getName())); //$NON-NLS-1$
        }
    }
}
