/*
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "{}"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright $date.year Alan Tai
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package android.lib.database;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import android.content.Context;
import android.database.Cursor;
import android.database.SQLException;
import android.database.sqlite.SQLiteDatabase;
import android.lib.database.predicate.Predicate;
import android.lib.database.query.Delete;
import android.lib.database.query.Insert;
import android.lib.database.query.Query;
import android.lib.database.query.Select;
import android.lib.database.query.Update;
import android.text.TextUtils;

/**
 * Provides database abstraction layer APIs to handle SQLite operations: {@code SELECT},
 * {@code UPDATE}, {@code INSERT}, {@code DELETE}.
 */
public class Database {
    private final SQLiteDatabase database;

    protected Database(final SQLiteDatabase database) {
        this.database = database;
    }

    /**
     * Creates a new instance of {@link Database} with the given {@code context},
     * {@code name}, {@code version}.
     * @param context the context to use to open or create the database.
     * @param name the file name of the database file, or null for an in-memory database.
     * @param version the version number of the database (starting at 1); if the database is older,
     * {@link DatabaseOpenHelper#onUpgrade(SQLiteDatabase, int, int)} will be used to upgrade the database;
     * if the database is newer, {@link DatabaseOpenHelper#onDowngrade(SQLiteDatabase, int, int)} will be used to downgrade the database.
     * @param tables the tables of the database to create in when the application runs for the first time.
     * For subsequent call, {@code tables} can be {@code null}.
     */
    protected Database(final Context context, final String name, final int version, final Class<?>... tables) {
        this(DatabaseOpenHelper.newInstance(context, name, version, tables).getWritableDatabase());
    }

    /**
     * Creates a new instance of {@link Database} with the given {@code context}, {@code name}.
     * @param context the context to use to open or create the database.
     * @param name the file name of the database file, or null for an in-memory database.
     * @param tables the tables of the database to create in when the application runs for the first time.
     * For subsequent call, {@code tables} can be {@code null}.
     */
    protected Database(final Context context, final String name, final Class<?>... tables) {
        this(DatabaseOpenHelper.newInstance(context, name, tables).getWritableDatabase());
    }

    /**
     * Creates a new instance of {@link Database} with the given {@code context}, {@code version}.
     * @param context the context to use to open or create the database.
     * @param version the version number of the database (starting at 1); if the database is older,
     * {@link DatabaseOpenHelper#onUpgrade(SQLiteDatabase, int, int)} will be used to upgrade the database;
     * if the database is newer, {@link DatabaseOpenHelper#onDowngrade(SQLiteDatabase, int, int)} will be used to downgrade the database.
     * @param tables the tables of the database to create in when the application runs for the first time.
     * For subsequent call, {@code tables} can be {@code null}.
     */
    protected Database(final Context context, final int version, final Class<?>... tables) {
        this(DatabaseOpenHelper.newInstance(context, version, tables).getWritableDatabase());
    }

    /**
     * Creates a new instance of {@link Database} with the given {@code context}.
     * @param context the context to use to open or create the database.
     * {@link DatabaseOpenHelper#onUpgrade(SQLiteDatabase, int, int)} will be used to upgrade the database;
     * if the database is newer, {@link DatabaseOpenHelper#onDowngrade(SQLiteDatabase, int, int)} will be used to downgrade the database.
     * @param tables the tables of the database to create in when the application runs for the first time.
     * For subsequent call, {@code tables} can be {@code null}.
     */
    protected Database(final Context context, final Class<?>... tables) {
        this(DatabaseOpenHelper.newInstance(context, tables).getWritableDatabase());
    }

    /**
     * Creates a new instance of {@link Database} with the given {@code context},
     * {@code name}, {@code version}.
     * @param context the context to use to open or create the database.
     * @param name the file name of the database file, or null for an in-memory database.
     * @param version the version number of the database (starting at 1); if the database is older,
     * {@link DatabaseOpenHelper#onUpgrade(SQLiteDatabase, int, int)} will be used to upgrade the database;
     * if the database is newer, {@link DatabaseOpenHelper#onDowngrade(SQLiteDatabase, int, int)} will be used to downgrade the database.
     * @param tables the tables of the database to create in when the application runs for the first time.
     * For subsequent call, {@code tables} can be {@code null}.
     */
    public static Database newInstance(final Context context, final String name, final int version, final Class<?>... tables) {
        return new Database(context, name, version, tables);
    }

    /**
     * Creates a new instance of {@link Database} with the given {@code context}, {@code name}.
     * @param context the context to use to open or create the database
     * @param name the file name of the database file, or null for an in-memory database.
     * @param tables the tables of the database to create in when the application runs for the first time.
     * For subsequent call, {@code tables} can be {@code null}.
     */
    public static Database newInstance(final Context context, final String name, final Class<?>... tables) {
        return new Database(context, name, tables);
    }

    /**
     * Creates a new instance of {@link Database} with the given {@code context}, {@code version}.
     * @param context the context to use to open or create the database.
     * @param version the version number of the database (starting at 1); if the database is older,
     * {@link DatabaseOpenHelper#onUpgrade(SQLiteDatabase, int, int)} will be used to upgrade the database;
     * if the database is newer, {@link DatabaseOpenHelper#onDowngrade(SQLiteDatabase, int, int)} will be used to downgrade the database.
     * @param tables the tables of the database to create in when the application runs for the first time.
     * For subsequent call, {@code tables} can be {@code null}.
     */
    public static Database newInstance(final Context context, final int version, final Class<?>... tables) {
        return new Database(context, version, tables);
    }

    /**
     * Creates a new instance of {@link Database} with the given {@code context}.
     * @param context the context to use to open or create the database.
     * @param tables the tables of the database to create in when the application runs for the first time.
     * For subsequent call, {@code tables} can be {@code null}.
     */
    public static Database newInstance(final Context context, final Class<?>... tables) {
        return new Database(context, tables);
    }

    /**
     * Creates database tables/columns/indices.
     * <p>Call this method on application first-run after installation.</p>
     * <p>This method should be called once per application installation.</p>
     * @param context the context to use to open the database.
     * @param tables the tables of the database to create.
     */
    public static void createTables(final Context context, final Class<?>... tables) {
        Database.newInstance(context, tables).close();
    }

    /**
     * Closes the database, and abort any uncommitted changes.
     * <p>This is a no-op if the database has already been closed before.</p>
     */
    public void close() {
        if (this.database.isOpen()) {
            if (this.database.inTransaction()) {
                this.database.endTransaction();
            }

            this.database.close();
        }
    }

    /**
     * Persists any uncommitted changes to the database.
     * <p>The current transaction, if any, will be closed when this method returns.
     * If necessary, call {@link #beginTransaction()} to start a new transaction.</p>
     * <p>This is no-op if there is no uncommitted changes</p>
     * @throws IllegalStateException if the database has been closed.
     */
    public void commit() {
        if (this.database.isOpen()) {
            if (this.database.inTransaction()) {
                this.database.setTransactionSuccessful();
                this.database.endTransaction();
            }
        } else {
            throw new IllegalStateException("Database has been closed"); //$NON-NLS-1$
        }
    }

    /**
     * Rolls back any uncommitted changes of the database.
     * <p>The current transaction, if any, will be closed when this method returns.
     * If necessary, call {@link #beginTransaction()} to start a new transaction.</p>
     * <p>This is a no-op if there is no uncommitted changes.</p>
     * @throws IllegalStateException if the database has been closed.
     */
    public void rollback() {
        if (this.database.isOpen()) {
            if (this.database.inTransaction()) {
                this.database.endTransaction();
            }
        } else {
            throw new IllegalStateException("Database has been closed"); //$NON-NLS-1$
        }
    }

    /**
     * Starts a new transaction.
     * <p>If a transaction has already been started and not closed yet,
     * a nested transaction will be started.</p>
     * @throws IllegalStateException if the database has been closed.
     */
    public void beginTransaction() {
        if (this.database.isOpen()) {
            this.database.beginTransaction();
        } else {
            throw new IllegalStateException("Database has been closed"); //$NON-NLS-1$
        }
    }

    /**
     * Executes a {@link Query} of type {@link Select} against the database and returns a list of object
     * of type {@code T} using the given {@link RowMapper}.
     * @param query the {@link Query} object containing
     * the raw SQL statement to be executed.
     * @param mapper the mapper to map {@link Cursor} to {@code T}.
     * @param <T> the type of the table object to cast the result set to.
     * @return the result set returned from the query execution.
     * @throws IllegalArgumentException if the column name is not found from the resulting <code>cursor</code>.
     * @throws UnsupportedTypeException if the type {@code T} contains fields of unsupported types.
     * @throws IllegalAccessException if the if the default constructor of {@code T} is not visible,
     * or the default constructor of the {@link TypeConverter} associated with any field of {@code T} is not visible.
     * @throws InstantiationException if the instance of {@code T} can not be created,
     * or the instance of the {@link TypeConverter} associated with any field of {@code T} can not be created.
     * @throws SQLException if the SQL string is invalid.
     */
    public <T> List<T> execute(final Query query, final RowMapper<T> mapper) throws IllegalAccessException, InstantiationException, UnsupportedTypeException, SQLException {
        final List<T> rows   = new ArrayList<>();
        final Cursor  cursor = this.database.rawQuery(query.getRawSQL(), Database.getParameters(query));

        while (cursor.moveToNext()) {
            rows.add(mapper.mapRow(cursor));
        }

        return rows;
    }

    /**
     * Execute SQL <code>SELECT</code> statement against the database and returns the row matching
     * the given <code>id</code>, or {@code null} if there is no matching row.
     * @param table the table containing the data to retrieve from.
     * @param idColumn the name of the id column.
     * @param idValue an unique value to match the id column of a row.
     * @return the result set returned from the query execution,
     * or {@code null} if there is no matching row.
     * @throws IllegalArgumentException if the column name is not found from the resulting <code>cursor</code>.
     * @throws UnsupportedTypeException if the type {@code T} contains fields of unsupported types.
     * @throws IllegalAccessException if the if the default constructor of {@code T} is not visible,
     * or the default constructor of the {@link TypeConverter} associated with any field of {@code T} is not visible.
     * @throws InstantiationException if the instance of {@code T} can not be created,
     * or the instance of the {@link TypeConverter} associated with any field of {@code T} can not be created.
     * @throws SQLException if the SQL string is invalid.
     */
    public <T> T selectById(final Class<T> table, final String idColumn, final int idValue) throws IllegalAccessException, InstantiationException, UnsupportedTypeException, SQLException {
        final List<T> rows = this.execute(Query.select().columns().from(table).where(Predicate.equalTo(idColumn, idValue, false, true)).build(), new RowMapper<>(table));

        if (rows.isEmpty()) {
            return null;
        }

        return rows.get(0);
    }

    /**
     * Executes SQL <code>INSERT</code> statement against the database.
     * @param row the data to insert into the database.
     * @throws UnsupportedTypeException if the type {@code T} contains fields of unsupported types.
     * @throws IllegalAccessException if the if the default constructor of {@code T} is not visible,
     * or the default constructor of the {@link TypeConverter} associated with any field of {@code T} is not visible.
     * @throws InstantiationException if the instance of {@code T} can not be created,
     * or the instance of the {@link TypeConverter} associated with any field of {@code T} can not be created.
     * @throws SQLException if the SQL string is invalid.
     */
    public <T> void insert(final T row) throws IllegalAccessException, InstantiationException, UnsupportedTypeException, SQLException {
        final Class<T>   clazz  = (Class<T>)row.getClass();
        final Insert     insert = Query.insert().into(clazz);
        final Set<Field> fields = new HashSet<>();

        for (final Field field : clazz.getFields()) {
            fields.add(field);
        }

        for (final Field field : clazz.getDeclaredFields()) {
            fields.add(field);
        }

        for (final Field field : fields) {
            field.setAccessible(true);

            final Column column = field.getAnnotation(Column.class);

            if (column != null && !column.autoIncrement()) {
                insert.set(TextUtils.isEmpty(column.value()) ? field.getName() : column.value(), Database.getColumnValue(row, field));
            }
        }

        this.execute(insert.build());
    }

    /**
     * Execute SQL <code>DELETE</code> statement against the database and deletes any rows with
     * the matching id value.
     * @param table the table to delete rows from.
     * @param idColumn the name of the id column.
     * @param idValue an unique value to match the id column of a row.
     * @throws SQLException if the SQL string is invalid .
     */
    public <T> void deleteById(final Class<T> table, final String idColumn, final int idValue) throws SQLException {
        this.execute(Query.delete().from(table).where(Predicate.equalTo(idColumn, idValue, false, true)).build());
    }

    /**
     * Executes a {@link Query} of type {@link Update}, {@link Insert} or {@link Delete} against the database.
     * @param query the {@link Query} object to execute.
     * @throws SQLException if the SQL string is invalid .
     */
    public void execute(final Query query) throws SQLException {
        final String[] parameters = Database.getParameters(query);

        if (parameters == null) {
            this.database.execSQL(query.getRawSQL());
        } else {
            this.database.execSQL(query.getRawSQL(), parameters);
        }
    }

    private static String[] getParameters(final Query query) {
        final String[] parameters = query.getParameters().isEmpty() ? null : new String[query.getParameters().size()];

        if (parameters != null) {
            for (int i = query.getParameters().size(); --i >= 0;) {
                final Object parameter = query.getParameters().get(i);

                parameters[i] = parameter == null ? null : query.getParameters().get(i).toString();
            }
        }

        return parameters;
    }

    private static <T> Object getColumnValue(final T row, final Field field)
                    throws UnsupportedTypeException,
                    IllegalAccessException,
                    InstantiationException {
        final Class<?>     type      = field.getType();
        final UseConverter converter = field.getAnnotation(UseConverter.class);

        if (converter == null) {
            if (type.equals(boolean.class) || type.equals(Boolean.class)) {
                return field.get(row);
            } else if (type.equals(byte.class) || type.equals(Byte.class)) {
                return field.get(row);
            } else if (type.equals(byte[].class) || type.equals(Byte[].class)) {
                return field.get(row);
            } else if (type.equals(double.class) || type.equals(Double.class)) {
                return field.get(row);
            } else if (type.equals(float.class) || type.equals(Float.class)) {
                return field.get(row);
            } else if (type.equals(int.class) || type.equals(Integer.class)) {
                return field.get(row);
            } else if (type.equals(long.class) || type.equals(Long.class)) {
                return field.get(row);
            } else if (type.equals(short.class) || type.equals(Short.class)) {
                return field.get(row);
            } else if (type.equals(String.class)) {
                return field.get(row);
            } else {
                throw new UnsupportedTypeException(String.format("%s is not of SQLite-compatible type", field.getName())); //$NON-NLS-1$
            }
        }

        final Class<?> fromType = converter.type();

        if (fromType.equals(boolean.class) || fromType.equals(Boolean.class)) {
            return ((TypeConverter<?, T>)converter.value().newInstance()).toDatabase((T)field.get(row));
        } else if (fromType.equals(byte.class) || fromType.equals(Byte.class)) {
            return ((TypeConverter<?, T>)converter.value().newInstance()).toDatabase((T)field.get(row));
        } else if (fromType.equals(byte[].class)) {
            return ((TypeConverter<?, T>)converter.value().newInstance()).toDatabase((T)field.get(row));
        } else if (fromType.equals(Byte[].class)) {
            return ((TypeConverter<?, T>)converter.value().newInstance()).toDatabase((T)field.get(row));
        } else if (fromType.equals(double.class) || fromType.equals(Double.class)) {
            return ((TypeConverter<?, T>)converter.value().newInstance()).toDatabase((T)field.get(row));
        } else if (fromType.equals(float.class) || fromType.equals(Float.class)) {
            return ((TypeConverter<?, T>)converter.value().newInstance()).toDatabase((T)field.get(row));
        } else if (fromType.equals(int.class) || fromType.equals(Integer.class)) {
            return ((TypeConverter<?, T>)converter.value().newInstance()).toDatabase((T)field.get(row));
        } else if (fromType.equals(long.class) || fromType.equals(Long.class)) {
            return ((TypeConverter<?, T>)converter.value().newInstance()).toDatabase((T)field.get(row));
        } else if (fromType.equals(short.class) || fromType.equals(Short.class)) {
            return ((TypeConverter<?, T>)converter.value().newInstance()).toDatabase((T)field.get(row));
        } else if (fromType.equals(String.class)) {
            return ((TypeConverter<?, T>)converter.value().newInstance()).toDatabase((T)field.get(row));
        } else {
            throw new UnsupportedTypeException(String.format("%s is not of SQLite-compatible type", field.getName())); //$NON-NLS-1$
        }
    }
}
