package android.dev.support.recycler.adapter.caller;

import android.dev.support.recycler.adapter.RecyclerAdapter;
import android.dev.support.recycler.adapter.SuccinctViewAdapter;
import android.dev.support.recycler.adapter.binder.AbstractBinder;
import android.dev.support.recycler.adapter.binder.ViewTypeBinder;
import android.dev.support.recycler.adapter.creator.ViewTypeCreator;
import android.dev.support.recycler.adapter.holder.ViewHolderMonitor;
import android.dev.support.recycler.adapter.holder.ViewTypeHolder;
import android.dev.support.recycler.adapter.listener.ViewEventPerformer;
import android.support.annotation.CallSuper;
import android.support.annotation.LayoutRes;
import android.support.annotation.NonNull;
import android.util.SparseArray;
import android.view.ViewGroup;

import java.util.List;

/**
 * Created by wbs on 2017/8/30 0030.
 */

public abstract class SuccinctPool<D, Holder extends ViewTypeHolder> {

    public final static int UN_KNOWN_VIEW_TYPE = -1;

    private final SparseArray<AbstractBinder<D, Holder>> mTypes = new SparseArray<>();

    private AbstractBinder<D, Holder> mUnknownViewType;

    private final TypeDependent<D, Holder> mTypeDependent;

    public SuccinctPool() {
        this(null);
    }

    public SuccinctPool(TypeDependent<D, Holder> typeDependent) {
        mTypeDependent = typeDependent == null ? getDefaultTypeDependent() : typeDependent;
    }

    private TypeDependent<D, Holder> getDefaultTypeDependent() {
        return new TypeDependent<D, Holder>() {
            @Override
            public int getItemViewType(RecyclerAdapter<D> adapter, SuccinctPool<D, Holder> pool, int position) {
                for (int i = 0; i < mTypes.size(); i++) {
                    if (mTypes.valueAt(i).typeOf(adapter, position)) {
                        return mTypes.keyAt(i);
                    }
                }
                return UN_KNOWN_VIEW_TYPE;
            }
        };
    }

    public SuccinctPool<D, Holder> unregister(int key) {
        mTypes.remove(key);
        return this;
    }

    public SuccinctPool<D, Holder> reset() {
        mTypes.clear();
        return this;
    }

    public SuccinctPool<D, Holder> unableUnknownViewType() {
        mUnknownViewType = null;
        return this;
    }

    public SuccinctPool<D, Holder> enableUnknownViewType() {
        mUnknownViewType = new ViewTypeBinder<>(provideUnknownViewTypeCreator());
        return this;
    }

    public SuccinctPool<D, Holder> enableUnknownViewType(AbstractBinder<D, Holder> binder) {
        mUnknownViewType = binder;
        return this;
    }

    @NonNull
    public Holder onCreateViewHolder(RecyclerAdapter<D> adapter, ViewGroup parent, int viewType) {

        AbstractBinder<D, Holder> type = mTypes.get(viewType);
        Holder vh;
        if (type == null) {
            if (mUnknownViewType == null) {
                throw new NullPointerException("No ViewTypeBinder registered for ViewType " + viewType);
            }
            vh = mUnknownViewType.onCreateViewHolder(adapter, parent);
        } else {
            vh = type.onCreateViewHolder(adapter, parent);
        }
        if (vh == null) {
            throw new NullPointerException(
                    "ViewHolder returned from " + type + " for ViewType =" + viewType
                            + " is null!");
        }
        return vh;
    }


    public void onBindViewHolder(RecyclerAdapter<D> adapter, Holder holder, int position, List<Object> payloads) {
        int viewType = holder.getItemViewType();
        AbstractBinder<D, Holder> typeBinder = mTypes.get(viewType);
        if (typeBinder == null) {
            if (mUnknownViewType == null) {
                throw new NullPointerException("No ViewTypeBinder registered for ViewType " + viewType);
            }
            mUnknownViewType.onBindViewHolder(adapter, position, holder, payloads);
            return;
        }
        typeBinder.onBindViewHolder(adapter, position, holder, payloads);
    }

    public int getItemViewType(RecyclerAdapter<D> adapter, int position) {
        return mTypeDependent.getItemViewType(adapter, this, position);
    }

    public void onAttachToWindow(RecyclerAdapter<D> adapter, Holder holder) {
        int viewType = holder.getItemViewType();
        AbstractBinder<D, Holder> typeBinder = mTypes.get(viewType);
        if (typeBinder == null) {
            if (mUnknownViewType == null) {
                throw new NullPointerException("No ViewTypeBinder registered for ViewType " + viewType);
            }
            mUnknownViewType.onAttachToWindow(adapter, holder);
            return;
        }
        typeBinder.onAttachToWindow(adapter, holder);
    }

    public void onDetachFromWindow(RecyclerAdapter<D> adapter, Holder holder) {
        int viewType = holder.getItemViewType();
        AbstractBinder<D, Holder> typeBinder = mTypes.get(viewType);
        if (typeBinder == null) {
            if (mUnknownViewType != null) {
                mUnknownViewType.onDetachFromWindow(adapter, holder);
            }
            return;
        }
        typeBinder.onDetachFromWindow(adapter, holder);
    }

    public void onUnbindViewHolder(RecyclerAdapter<D> adapter, Holder holder) {
        int viewType = holder.getItemViewType();
        AbstractBinder<D, Holder> typeBinder = mTypes.get(viewType);
        if (typeBinder == null) {
            if (mUnknownViewType != null) {
                mUnknownViewType.onUnbindViewHolder(adapter, holder);
            }
            return;
        }
        typeBinder.onUnbindViewHolder(adapter, holder);
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(AbstractBinder<D, Holder> binder) {
        int size = mTypes.size();
        mTypes.put(size, binder);
        return this;
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(@LayoutRes int layoutId) {
        return register(layoutId, null, null, null, null);
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(@LayoutRes int layoutId, TypeDecider<D> decider) {
        return register(layoutId, decider, null, null, null);
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(@LayoutRes int layoutId, TypeDecider<D> decider, ViewInjector<D, Holder> injector) {
        return register(layoutId, decider, null, injector, null);
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(@LayoutRes int layoutId, ViewEventPerformer<Holder> performer) {
        return register(layoutId, null, performer, null, null);
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(@LayoutRes int layoutId, TypeDecider<D> decider, ViewEventPerformer<Holder> performer) {
        return register(layoutId, decider, performer, null, null);
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(@LayoutRes int layoutId, ViewInjector<D, Holder> injector) {
        return register(layoutId, null, null, injector, null);
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(@LayoutRes int layoutId, ViewEventPerformer<Holder> performer, ViewInjector<D, Holder> injector) {
        return register(layoutId, null, performer, injector, null);
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(@LayoutRes int layoutId, ViewTypeInjector<D, Holder> injector) {
        return register(layoutId, injector, null, injector, null);
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(@LayoutRes int layoutId, ViewEventPerformer<Holder> performer, ViewTypeInjector<D, Holder> injector) {
        return register(layoutId, injector, performer, injector, null);
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(@LayoutRes int viewTypeLayout, ViewTypeInjector<D, Holder> viewTypeInjector, ViewHolderMonitor<D, Holder> monitor) {
        return register(viewTypeLayout, viewTypeInjector, null, viewTypeInjector, monitor);
    }

    public abstract SuccinctPool<D, Holder> register(@LayoutRes int layoutId, TypeDecider<D> decider, ViewEventPerformer<Holder> performer, ViewInjector<D, Holder> injector, ViewHolderMonitor<D, Holder> monitor);

    @CallSuper
    public SuccinctPool<D, Holder> register(ViewTypeCreator<Holder> creator) {
        return register(creator, null, null, null, null);
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(ViewTypeCreator<Holder> creator, TypeDecider<D> decider) {
        return register(creator, decider, null, null, null);
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(ViewTypeCreator<Holder> creator, TypeDecider<D> decider, ViewInjector<D, Holder> injector) {
        return register(creator, null, null, injector, null);
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(ViewTypeCreator<Holder> creator, ViewEventPerformer<Holder> performer) {
        return register(creator, null, performer, null, null);
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(ViewTypeCreator<Holder> creator, TypeDecider<D> decider, ViewEventPerformer<Holder> performer) {
        return register(creator, null, performer, null, null);
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(ViewTypeCreator<Holder> creator, ViewInjector<D, Holder> injector) {
        return register(creator, null, null, injector, null);
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(ViewTypeCreator<Holder> creator, ViewEventPerformer<Holder> performer, ViewInjector<D, Holder> injector) {
        return register(creator, null, performer, injector, null);
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(ViewTypeCreator<Holder> creator, ViewTypeInjector<D, Holder> injector) {
        return register(creator, injector, null, injector, null);
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(ViewTypeCreator<Holder> creator, ViewEventPerformer<Holder> performer, ViewTypeInjector<D, Holder> injector) {
        return register(creator, injector, performer, injector, null);
    }

    @CallSuper
    public SuccinctPool<D, Holder> register(ViewTypeCreator<Holder> creator, TypeDecider<D> decider, ViewEventPerformer<Holder> performer, ViewInjector<D, Holder> injector, ViewHolderMonitor<D, Holder> monitor) {
        ViewTypeBinder<D, Holder> binder = new ViewTypeBinder<>(decider, creator, performer, injector, monitor);
        register(binder);
        return this;
    }

    public abstract SuccinctViewAdapter<D, Holder> adapt();


    protected abstract ViewTypeCreator<Holder> provideUnknownViewTypeCreator();
}
