package android.dev.support.recycler.adapter;

import android.dev.support.recycler.adapter.caller.RecyclerItemClickListener;
import android.dev.support.recycler.adapter.caller.SuccinctPool;
import android.dev.support.recycler.adapter.caller.ViewHolderPerformAction;
import android.dev.support.recycler.adapter.holder.ViewTypeHolder;
import android.support.v7.util.DiffUtil;
import android.support.v7.widget.RecyclerView;
import android.view.View;
import android.view.ViewGroup;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by wbs on 2017/9/1 0001.
 */

public abstract class RecyclerViewAdapter<T, Holder extends ViewTypeHolder> extends
        RecyclerView.Adapter<Holder> implements RecyclerAdapter<T> {

    final List<T> mData = new ArrayList<>();

    private RecyclerItemClickListener mItemClickedListener;

    private ViewHolderPerformAction<T, Holder> mPerformAction;

    @Override
    public final Holder onCreateViewHolder(ViewGroup parent, int viewType) {
        final Holder holder = createItemViewHolder(parent, viewType);
        if (mPerformAction != null) {
            mPerformAction.setViewHolderAction(this, holder, viewType);
        }
        if (mItemClickedListener != null && validViewType(viewType)) {
            getItemViewRoot(holder).setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    mItemClickedListener.onItemClicked(RecyclerViewAdapter.this, holder, holder.getLayoutPosition());
                }
            });
        }
        return holder;
    }

    protected abstract Holder createItemViewHolder(ViewGroup parent, int viewType);

    @Override
    public final void onBindViewHolder(ViewTypeHolder holder, int position) {

    }

    @Override
    public int getItemCount() {
        return mData.size();
    }

    @Override
    public T getItem(int position) {
        return mData.get(position);
    }

    @Override
    public void add(T item) {
        add(mData.size(), item);
    }

    @Override
    public void add(int position, T item) {
        mData.add(position, item);
        notifyItemInserted(position);
    }

    @Override
    public void remove(T item) {
        remove(mData.indexOf(item));
    }

    @Override
    public void remove(int position) {
        if (position < 0) {
            return;
        }
        mData.remove(position);
        notifyItemRemoved(position);
    }

    @Override
    public void refreshData(List<? extends T> data) {
        if (data == null) {
            clear();
            return;
        }
        mData.clear();
        mData.addAll(data);
        notifyDataSetChanged();
    }

    @Override
    public void clear() {
        mData.clear();
        notifyDataSetChanged();
    }

    @Override
    public void addData(List<? extends T> data) {
        if (data != null && !data.isEmpty()) {
            mData.addAll(data);
            int newSize = data.size();
            int position = mData.size() - newSize;
            notifyItemRangeInserted(position, newSize);
        }
    }


    @Override
    public List<T> getCollections() {
        return mData;
    }

    public RecyclerViewAdapter<T, Holder> setItemClickListener(RecyclerItemClickListener listener) {
        mItemClickedListener = listener;
        return this;
    }

    public void registerViewHolderPerformAction(ViewHolderPerformAction<T, Holder> action) {
        mPerformAction = action;
    }

    protected boolean validViewType(int viewType) {
        return viewType != SuccinctPool.UN_KNOWN_VIEW_TYPE && viewType >= 0 && viewType < Integer.MAX_VALUE;
    }

    protected View getItemViewRoot(ViewTypeHolder holder) {
        return holder.itemView;
    }

    public void refreshData(List<? extends T> data, DiffUtil.Callback callback) {
        DiffUtil.DiffResult diffResult = DiffUtil.calculateDiff(callback, false);
        mData.clear();
        mData.addAll(data);
        diffResult.dispatchUpdatesTo(this);
    }

}
