package android.dev.support.recycler.ex;

import android.content.Context;
import android.support.annotation.Nullable;
import android.support.v7.widget.RecyclerView;
import android.support.v7.widget.StaggeredGridLayoutManager;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewGroup;

import java.util.ArrayList;

/**
 * Created on 2018/6/20
 *
 * @author wenbosheng
 * Copyright (c) 2018 Amber
 */
public class RecyclerViewPlus extends RecyclerView {

    private WrapAdapter mAdapter;

    public RecyclerViewPlus(Context context) {
        super(context);
    }

    public RecyclerViewPlus(Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
    }

    public RecyclerViewPlus(Context context, @Nullable AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
    }

    @Override
    public void setAdapter(Adapter adapter) {
        super.setAdapter((mAdapter = new WrapAdapter(adapter)));
    }

    @Override
    public Adapter getAdapter() {
        if (mAdapter != null) {
            return mAdapter.getInnerAdapter();
        }
        return super.getAdapter();
    }

    public void addHeader(View header) {
        if (mAdapter != null) {
            mAdapter.addHeaderView(header);
        }
    }

    public void addFooter(View footer) {
        if (mAdapter != null) {
            mAdapter.addFooterView(footer);
        }
    }

    public View getHeaderView() {
        if (mAdapter == null) return null;
        return mAdapter.getHeaderView();
    }

    public View getFooterView(){
        if (mAdapter == null) return null;
        return mAdapter.getFooterView();
    }

    public int getHeaderViewSize(){
        if(mAdapter==null)return 0;
        return mAdapter.getHeaderViewsCount();
    }

    public int getFooterViewSize(){
        if(mAdapter==null)return 0;
        return mAdapter.getFooterViewsCount();
    }

    private static class WrapAdapter extends RecyclerView.Adapter<RecyclerView.ViewHolder> {

        private final static int MAX_COUNT = (Integer.MIN_VALUE / 2 - Integer.MIN_VALUE) / 2;
        //decide min = -2147483648;

        //make header view type values  in min~~min+MAX_COUNT
        private static final int TYPE_HEADER_VIEW_MIN = Integer.MIN_VALUE;

        private static final int TYPE_HEADER_VIEW_MAX = Integer.MIN_VALUE + MAX_COUNT;

        //make footer view type values int min+MAX_COUNT~~min+MAX_COUNT*2
        private static final int TYPE_FOOTER_VIEW_MIN = TYPE_HEADER_VIEW_MAX + 1;

        private static final int TYPE_FOOTER_VIEW_MAX = Integer.MIN_VALUE + MAX_COUNT * 2;

        private RecyclerView.Adapter mInnerAdapter;

        private final ArrayList<View> mHeaderViews = new ArrayList<>();

        private final ArrayList<View> mFooterViews = new ArrayList<>();

        private RecyclerView mRecyclerView;

        private RecyclerView.LayoutManager mLayoutManager;

        public WrapAdapter(RecyclerView.Adapter adapter) {
            mInnerAdapter = adapter;
        }

        @Override
        public int getItemViewType(int position) {
            int innerCount = mInnerAdapter.getItemCount();
            int headerViewsSize = getHeaderViewsCount();
            if (position < headerViewsSize) {
                return TYPE_HEADER_VIEW_MIN + position;
            } else if (headerViewsSize <= position && position < headerViewsSize + innerCount) {

                int innerItemViewType = mInnerAdapter.getItemViewType(position - headerViewsSize);
                if (innerItemViewType <= TYPE_FOOTER_VIEW_MAX) {
                    throw new IllegalArgumentException("your adapter's return value of getItemViewType() must > " + TYPE_FOOTER_VIEW_MAX);
                }
                return innerItemViewType;
            } else {
                int footerIndex = position - headerViewsSize - innerCount;
                return TYPE_FOOTER_VIEW_MIN + footerIndex;
            }
        }

        @Override
        public RecyclerView.ViewHolder onCreateViewHolder(ViewGroup parent, int viewType) {
            if (viewType <= TYPE_HEADER_VIEW_MAX) {
                return new ViewHolder(mHeaderViews.get(viewType - TYPE_HEADER_VIEW_MIN)) {
                };
            } else if (viewType <= TYPE_FOOTER_VIEW_MAX) {
                return new ViewHolder(mFooterViews.get(viewType - TYPE_FOOTER_VIEW_MIN)) {
                };
            } else {
                return mInnerAdapter.onCreateViewHolder(parent, viewType);
            }
        }

        @Override
        public void onBindViewHolder(RecyclerView.ViewHolder holder, int position) {
            int viewType = holder.getItemViewType();
            if (viewType <= TYPE_FOOTER_VIEW_MAX) {
                ViewGroup.LayoutParams layoutParams = holder.itemView.getLayoutParams();

                if (layoutParams instanceof StaggeredGridLayoutManager.LayoutParams) {
                    ((StaggeredGridLayoutManager.LayoutParams) layoutParams).setFullSpan(true);
                } else if (layoutParams == null && mLayoutManager instanceof StaggeredGridLayoutManager) {
                    StaggeredGridLayoutManager.LayoutParams lp = new StaggeredGridLayoutManager.LayoutParams(RecyclerView.LayoutParams.MATCH_PARENT, RecyclerView.LayoutParams.WRAP_CONTENT);

                    lp.setFullSpan(true);
                    holder.itemView.setLayoutParams(lp);
                }
            } else {
                mInnerAdapter.onBindViewHolder(holder, position - getHeaderViewsCount());
            }
        }

        @Override
        public int getItemCount() {
            return getHeaderViewsCount() + getFooterViewsCount() + mInnerAdapter.getItemCount();
        }

        @Override
        public void onAttachedToRecyclerView(RecyclerView recyclerView) {
            mRecyclerView = recyclerView;
            mInnerAdapter.onAttachedToRecyclerView(recyclerView);

            if (mLayoutManager == null) {
                mLayoutManager = recyclerView.getLayoutManager();
            }
        }

        @Override
        public void onViewAttachedToWindow(RecyclerView.ViewHolder holder) {
            super.onViewAttachedToWindow(holder);

            if (mLayoutManager == null && mRecyclerView != null) {
                mLayoutManager = mRecyclerView.getLayoutManager();
            }
        }

        private final RecyclerView.AdapterDataObserver mDataObserver = new RecyclerView.AdapterDataObserver() {

            @Override
            public void onChanged() {
                super.onChanged();
                notifyDataSetChanged();
            }

            @Override
            public void onItemRangeChanged(int positionStart, int itemCount) {
                super.onItemRangeChanged(positionStart, itemCount);
                notifyItemRangeChanged(positionStart + getHeaderViewsCount(), itemCount);
            }

            @Override
            public void onItemRangeInserted(int positionStart, int itemCount) {
                super.onItemRangeInserted(positionStart, itemCount);
                notifyItemRangeInserted(positionStart + getHeaderViewsCount(), itemCount);
            }

            @Override
            public void onItemRangeRemoved(int positionStart, int itemCount) {
                super.onItemRangeRemoved(positionStart, itemCount);
                notifyItemRangeRemoved(positionStart + getHeaderViewsCount(), itemCount);
            }

            @Override
            public void onItemRangeMoved(int fromPosition, int toPosition, int itemCount) {
                super.onItemRangeMoved(fromPosition, toPosition, itemCount);
                int headerViewsCountCount = getHeaderViewsCount();
                notifyItemRangeChanged(fromPosition + headerViewsCountCount, toPosition + headerViewsCountCount + itemCount);
            }
        };

        //public methods
        public void setAdapter(RecyclerView.Adapter<RecyclerView.ViewHolder> adapter) {

            if (adapter == null) {
                return;
            }
            if (mInnerAdapter != null) {
                notifyItemRangeRemoved(getHeaderViewsCount(), mInnerAdapter.getItemCount());
                mInnerAdapter.unregisterAdapterDataObserver(mDataObserver);
            }
            this.mInnerAdapter = adapter;
            mInnerAdapter.registerAdapterDataObserver(mDataObserver);
            notifyItemRangeInserted(getHeaderViewsCount(), mInnerAdapter.getItemCount());
        }

        public RecyclerView.Adapter getInnerAdapter() {
            return mInnerAdapter;
        }

        public void addHeaderView(View header) {

            if (header == null) {
                throw new RuntimeException("header is null");
            }

            mHeaderViews.add(header);
            this.notifyDataSetChanged();
        }

        public void addFooterView(View footer) {

            if (footer == null) {
                throw new RuntimeException("footer is null");
            }

            mFooterViews.add(footer);
            this.notifyDataSetChanged();
        }

        public View getFooterView() {
            return getFooterViewsCount() > 0 ? mFooterViews.get(0) : null;
        }

        public View getHeaderView() {
            return getHeaderViewsCount() > 0 ? mHeaderViews.get(0) : null;
        }

        public void removeHeaderView(View view) {
            mHeaderViews.remove(view);
            this.notifyDataSetChanged();
        }

        public void removeFooterView(View view) {
            mFooterViews.remove(view);
            this.notifyDataSetChanged();
        }

        public int getHeaderViewsCount() {
            return mHeaderViews.size();
        }

        public int getFooterViewsCount() {
            return mFooterViews.size();
        }


    }

}
