package android.dev.support.recycler.gesture;

import android.graphics.Canvas;
import android.support.v7.widget.GridLayoutManager;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.support.v7.widget.helper.ItemTouchHelper;

/**
 * Created by oeager on 2016/11/22.
 */

public class DefaultItemTouchHelperCallback extends ItemTouchHelper.Callback {

    private final ItemTouchResultCallback mCallback;

    private ItemMovementProvider mProvider;

    private ItemEffectProvider mItemEffectProvider;

    private boolean mLongPressDragEnabled = false;

    private boolean mSwipeEnable = true;

    public DefaultItemTouchHelperCallback(ItemTouchResultCallback callback) {
        mCallback = callback;
    }

    public void setItemMovementProvider(ItemMovementProvider provider){
        mProvider = provider;
    }

    public void setItemEffectProvider(ItemEffectProvider provider){
        mItemEffectProvider = provider;
    }

    public void setSwipeEnable(boolean b){
        mSwipeEnable = b;
    }

    public void setLongPressDragEnabled(boolean b){
        mLongPressDragEnabled = b;
    }

    @Override
    public int getMovementFlags(RecyclerView recyclerView, RecyclerView.ViewHolder viewHolder) {
        if(mProvider==null){
            RecyclerView.LayoutManager layoutManager = recyclerView.getLayoutManager();
            if (layoutManager instanceof GridLayoutManager) {
                int dragFlags = ItemTouchHelper.UP | ItemTouchHelper.DOWN | ItemTouchHelper.LEFT | ItemTouchHelper.RIGHT;
                int swipeFlags = ItemTouchHelper.LEFT | ItemTouchHelper.RIGHT;
                return makeMovementFlags(dragFlags, swipeFlags);
            }
            if (layoutManager instanceof LinearLayoutManager) {
                LinearLayoutManager linearLayoutManager = (LinearLayoutManager) layoutManager;
                if (linearLayoutManager.getOrientation() == LinearLayoutManager.HORIZONTAL) {
                    int dragFlags = ItemTouchHelper.LEFT | ItemTouchHelper.RIGHT;
                    int swipeFlags = ItemTouchHelper.UP | ItemTouchHelper.DOWN;
                    return makeMovementFlags(dragFlags, swipeFlags);
                }
                int dragFlags = ItemTouchHelper.UP | ItemTouchHelper.DOWN;
                int swipeFlags = ItemTouchHelper.LEFT | ItemTouchHelper.RIGHT;
                return makeMovementFlags(dragFlags, swipeFlags);
            }
            return makeMovementFlags(0,0);
        }
        return makeMovementFlags(mProvider.onDragFlags(recyclerView,viewHolder),mProvider.onSwipeFlags(recyclerView,viewHolder));
    }

    @Override
    public boolean onMove(RecyclerView recyclerView, RecyclerView.ViewHolder srcHolder, RecyclerView.ViewHolder target) {
        if(mItemEffectProvider!=null&&mItemEffectProvider.requestViewHolderSameType()&&srcHolder.getItemViewType()!=target.getItemViewType()){
            return false;
        }
        if(mCallback==null){
            return false;
        }
        return mCallback.onItemMove(srcHolder.getAdapterPosition(),target.getAdapterPosition());
    }

    @Override
    public void onSwiped(RecyclerView.ViewHolder viewHolder, int direction) {
        if(mCallback!=null){
            mCallback.onItemDismiss(viewHolder.getAdapterPosition());
        }
    }

    @Override
    public void onSelectedChanged(RecyclerView.ViewHolder viewHolder, int actionState) {
        if(mItemEffectProvider !=null){
            mItemEffectProvider.onSelectedChanged(viewHolder,actionState);
        }
        super.onSelectedChanged(viewHolder, actionState);

    }

    @Override
    public void clearView(RecyclerView recyclerView, RecyclerView.ViewHolder viewHolder) {
        super.clearView(recyclerView, viewHolder);
        if(mItemEffectProvider !=null){
            mItemEffectProvider.onClearView(recyclerView,viewHolder);
        }
    }

    @Override
    public boolean isLongPressDragEnabled() {
        return mLongPressDragEnabled;
    }

    @Override
    public boolean isItemViewSwipeEnabled() {
        return mSwipeEnable;
    }

    @Override
    public void onChildDraw(Canvas c, RecyclerView recyclerView, RecyclerView.ViewHolder viewHolder, float dX, float dY, int actionState, boolean isCurrentlyActive) {
        super.onChildDraw(c, recyclerView, viewHolder, dX, dY, actionState, isCurrentlyActive);
        if(actionState== ItemTouchHelper.ACTION_STATE_SWIPE){
            //dx [-width,width] left->right:dx>0,right->left:dx<0
            if(mItemEffectProvider !=null){
                mItemEffectProvider.onSwipeChildDraw(c,recyclerView,viewHolder,dX,dY,actionState,isCurrentlyActive);
            }
        }
    }

    @Override
    public long getAnimationDuration(RecyclerView recyclerView, int animationType, float animateDx, float animateDy) {
        if(mItemEffectProvider==null){
            return super.getAnimationDuration(recyclerView,animationType,animateDx,animateDy);
        }
        long duration = mItemEffectProvider.getAnimationDuration(recyclerView,animationType,animateDx,animateDy);
        if(duration<0){
            return animationType == ItemTouchHelper.ANIMATION_TYPE_DRAG ? DEFAULT_DRAG_ANIMATION_DURATION
                    : DEFAULT_SWIPE_ANIMATION_DURATION;
        }
        return duration;
    }
    @Override
    public float getMoveThreshold(RecyclerView.ViewHolder viewHolder) {
        if(mItemEffectProvider==null){
            return super.getMoveThreshold(viewHolder);
        }
        float threshold = mItemEffectProvider.getMoveThreshold(viewHolder);
        return threshold>0?threshold:super.getMoveThreshold(viewHolder);
    }

    @Override
    public float getSwipeThreshold(RecyclerView.ViewHolder viewHolder) {
       if(mItemEffectProvider==null){
           return super.getSwipeThreshold(viewHolder);
       }
        float threshold =  mItemEffectProvider.getSwipeThreshold(viewHolder);
        return threshold>0?threshold:super.getMoveThreshold(viewHolder);
    }
}
