package android.dev.support.recycler.adapter.caller;

import android.dev.support.recycler.adapter.RecyclerAdapter;
import android.dev.support.recycler.adapter.SuccinctViewAdapter;
import android.dev.support.recycler.adapter.binder.UnknownViewTypeBinder;
import android.dev.support.recycler.adapter.binder.ViewTypeBinder;
import android.dev.support.recycler.adapter.creator.ViewTypeCreator;
import android.dev.support.recycler.adapter.holder.ViewHolderMonitor;
import android.dev.support.recycler.adapter.holder.ViewTypeHolder;
import android.support.annotation.LayoutRes;
import android.support.annotation.NonNull;
import android.util.SparseArray;
import android.view.ViewGroup;

import java.util.List;

/**
 * Created by wbs on 2017/8/30 0030.
 */

public abstract class SuccinctPool<D, Holder extends ViewTypeHolder> {

    private final SparseArray<ViewTypeBinder<D, Holder>> mTypes = new SparseArray<>();

    private UnknownViewTypeBinder<D, Holder> mUnknownViewType;

    private final TypeDependent<D, Holder> mTypeDependent;

    public SuccinctPool() {
        this(null);
    }

    public SuccinctPool(TypeDependent<D, Holder> typeDependent) {
        mTypeDependent = typeDependent == null ? getDefaultTypeDependent() : typeDependent;
    }

    private TypeDependent<D, Holder> getDefaultTypeDependent() {
        return new TypeDependent<D, Holder>() {
            @Override
            public int getItemViewType(RecyclerAdapter<D> adapter, SuccinctPool<D, Holder> pool, int position) {
                for (int i = 0; i < mTypes.size(); i++) {
                    if (mTypes.valueAt(i).typeOf(adapter, position)) {
                        return mTypes.keyAt(i);
                    }
                }
                return UnknownViewTypeBinder.UN_KNOWN_VIEW_TYPE;
            }
        };
    }

    public SuccinctPool<D, Holder> register(ViewTypeCreator<Holder> creator, ViewTypeInjector<D, Holder> viewTypeInjector) {
        ViewTypeBinder<D, Holder> binder = new ViewTypeBinder<>(viewTypeInjector, creator);
        return register(binder);
    }
    public SuccinctPool<D, Holder> register(ViewTypeCreator<Holder> creator, final TypeDecider<D> typeDecider) {
        return register(creator,typeDecider,null);
    }
    public SuccinctPool<D, Holder> register(ViewTypeCreator<Holder> creator, final TypeDecider<D> typeDecider, final ViewInjector<D, Holder> injector) {
        ViewTypeBinder<D, Holder> binder = new ViewTypeBinder<>(makeViewTypeInjector(typeDecider,injector), creator);
        return register(binder);
    }

    public SuccinctPool<D, Holder> register(ViewTypeCreator<Holder> creator, ViewTypeInjector<D, Holder> viewTypeInjector, ViewHolderMonitor<D, Holder> monitor) {
        ViewTypeBinder<D, Holder> binder = new ViewTypeBinder<>(viewTypeInjector, monitor, creator);
        return register(binder);
    }
    public SuccinctPool<D, Holder> register(ViewTypeCreator<Holder> creator,TypeDecider<D> typeDecider, ViewInjector<D, Holder> viewInjector, ViewHolderMonitor<D, Holder> monitor) {
        ViewTypeBinder<D, Holder> binder = new ViewTypeBinder<>(makeViewTypeInjector(typeDecider, viewInjector), monitor, creator);
        return register(binder);
    }
    public SuccinctPool<D, Holder> register(ViewTypeBinder<D, Holder> binder) {
        int size = mTypes.size();
        mTypes.put(size, binder);
        return this;
    }

    public SuccinctPool<D, Holder> unregister(int key) {
        mTypes.remove(key);
        return this;
    }

    public SuccinctPool<D, Holder> reset() {
        mTypes.clear();
        mUnknownViewType = null;
        return this;
    }

    public SuccinctPool<D, Holder> enableUnknownViewType() {
        mUnknownViewType = new UnknownViewTypeBinder<>(provideUnknownViewTypeCreator());
        return this;
    }

    public SuccinctPool<D, Holder> enableUnknownViewType(UnknownViewTypeBinder<D, Holder> binder) {
        mUnknownViewType = binder;
        return this;
    }

    @NonNull
    public Holder onCreateViewHolder(RecyclerAdapter<D> adapter, ViewGroup parent, int viewType) {

        ViewTypeBinder<D, Holder> type = mTypes.get(viewType);

        if (type == null) {
            if (mUnknownViewType == null) {
                throw new NullPointerException("No ViewTypeBinder registered for ViewType " + viewType);
            }
            return mUnknownViewType.onCreateViewHolder(adapter, parent);
        }

        Holder vh = type.onCreateViewHolder(adapter, parent);
        if (vh == null) {
            throw new NullPointerException(
                    "ViewHolder returned from " + type + " for ViewType =" + viewType
                            + " is null!");
        }
        return vh;
    }

    public SuccinctViewAdapter<D, Holder> adapt() {
        return new SuccinctViewAdapter<>(this);
    }

    public void onBindViewHolder(RecyclerAdapter<D> adapter, Holder holder, int position, List<Object> payloads) {
        int viewType = holder.getItemViewType();
        ViewTypeBinder<D, Holder> typeBinder = mTypes.get(viewType);
        if (typeBinder == null) {
            if (mUnknownViewType == null) {
                throw new NullPointerException("No ViewTypeBinder registered for ViewType " + viewType);
            }
            mUnknownViewType.onBindViewHolder(adapter, position, holder, payloads);
            return;
        }
        typeBinder.onBindViewHolder(adapter, position, holder, payloads);
    }

    public int getItemViewType(RecyclerAdapter<D> adapter, int position) {
        return mTypeDependent.getItemViewType(adapter, this, position);
    }

    public void onAttachToWindow(RecyclerAdapter<D> adapter, Holder holder) {
        int viewType = holder.getItemViewType();
        ViewTypeBinder<D, Holder> typeBinder = mTypes.get(viewType);
        if (typeBinder == null) {
            if (mUnknownViewType == null) {
                throw new NullPointerException("No ViewTypeBinder registered for ViewType " + viewType);
            }
            mUnknownViewType.onAttachToWindow(adapter, holder);
            return;
        }
        typeBinder.onAttachToWindow(adapter, holder);
    }

    public void onDetachFromWindow(RecyclerAdapter<D> adapter, Holder holder) {
        int viewType = holder.getItemViewType();
        ViewTypeBinder<D, Holder> typeBinder = mTypes.get(viewType);
        if (typeBinder == null) {
            if (mUnknownViewType == null) {
                throw new NullPointerException("No ViewTypeBinder registered for ViewType " + viewType);
            }
            mUnknownViewType.onDetachFromWindow(adapter, holder);
            return;
        }
        typeBinder.onDetachFromWindow(adapter, holder);
    }

    public void onUnbindViewHolder(RecyclerAdapter<D> adapter, Holder holder) {
        int viewType = holder.getItemViewType();
        ViewTypeBinder<D, Holder> typeBinder = mTypes.get(viewType);
        if (typeBinder == null) {
            if (mUnknownViewType == null) {
                throw new NullPointerException("No ViewTypeBinder registered for ViewType " + viewType);
            }
            mUnknownViewType.onUnbindViewHolder(adapter, holder);
            return;
        }
        typeBinder.onUnbindViewHolder(adapter, holder);
    }

    private ViewTypeInjector<D,Holder> makeViewTypeInjector(final TypeDecider<D> typeDecider, final ViewInjector<D,Holder> injector){
        return new ViewTypeInjector<D, Holder>() {
            @Override
            public boolean typeOf(RecyclerAdapter<D> adapter, int position) {
                if (typeDecider != null)
                    return typeDecider.typeOf(adapter, position);
                return true;
            }

            @Override
            public void onBindViewHolder(RecyclerAdapter<D> adapter, int position, @NonNull Holder holder, List<Object> payloads) {
                if (injector != null) {
                    injector.onBindViewHolder(adapter, position, holder, payloads);
                }
            }

            @Override
            public void onUnbindViewHolder(RecyclerAdapter<D> adapter, @NonNull Holder holder) {
                if (injector != null) {
                    injector.onUnbindViewHolder(adapter, holder);
                }
            }
        };
    }

    public abstract SuccinctPool<D, Holder> register(@LayoutRes int viewTypeLayout, ViewTypeInjector<D, Holder> viewTypeInjector);

    public abstract SuccinctPool<D, Holder> register(@LayoutRes int viewTypeLayout, ViewTypeInjector<D, Holder> viewTypeInjector, ViewHolderMonitor<D, Holder> monitor);

    protected abstract ViewTypeCreator<Holder> provideUnknownViewTypeCreator();
}
