package android.dev.decoupling

import org.gradle.api.NamedDomainObjectContainer
import org.gradle.api.Project

abstract class PluginWork {

    protected final static String EXTENSION_NAME = "dependencyOptions"

    protected final static String PROPERTY_MAIN_MODULAR = "main_modular"

    protected final static String PROPERTY_INDEPENDENT_MODULAR = "independent_modular"

    protected static boolean isDeBug = true

    protected abstract void apply(Project project)

    protected static void applyProjectLibrary(Project project){
        project.apply plugin: 'com.android.library'
    }

    protected static void applyProjectApplication(Project project,boolean isMainModule){
        project.apply plugin: 'com.android.application'
        if (!isMainModule) {
            project.android.sourceSets {
                main {
                    manifest.srcFile 'src/main/independent/AndroidManifest.xml'
                    java.srcDirs += ['src/main/independent/java']
                    res.srcDirs += ['src/main/independent/res']
                }
            }
        }
    }
    protected static void dumpDependable(StringBuilder builder, CompileBean dependable) {
        List<CompileCmd> dep = dependable.mBeans
        for (def cmd : dep) {
            def obj = cmd.dependency
            builder.append("\t\t").append(cmd.configuration).append(" ").append(obj.toString()).append("\n")
        }
    }

    protected static Flavors getFlavor(String taskName, DependencyExtension extension) {
        NamedDomainObjectContainer<Flavors> flavors = extension.flavors
        if (flavors == null || flavors.isEmpty()) {
            return null
        }
        for (def flavor : flavors) {
            if (taskName.toLowerCase().contains(flavor.name.toLowerCase())) {
                return flavor
            }
        }
        return null
    }

    protected static BuildTypes getBuildType(String taskName, DependencyExtension extension) {
        NamedDomainObjectContainer<BuildTypes> buildTypes = extension.buildTypes
        if (buildTypes == null || buildTypes.isEmpty()) {
            return null
        }
        for (def buildType : buildTypes) {
            if (taskName.toLowerCase().contains(buildType.name.toLowerCase())) {
                return buildType
            }
        }
        return null
    }

    protected static void dependCompile(Project project, List<CompileCmd> cmdList) {
        if (cmdList != null && !cmdList.isEmpty()) {
            for (def cmd : cmdList) {
                def dep = cmd.dependency
                int type = dep.getType()
                log("$cmd.configuration " + dep.toString() + " [$type]")
                def target = dep.getDepend(project)
                project.dependencies.add(cmd.configuration, target)
            }
        }

    }
    protected static void dumpUserConfiguration(DependencyExtension de){
        StringBuilder builder = new StringBuilder()
        builder.append("user configuration list:").append("\n")
        builder.append("flavors:{").append("\n")
        for (def flavor : de.flavors) {
            builder.append("\t").append(flavor.name).append("{").append("\n")
            dumpDependable(builder, flavor)
            builder.append("\t").append("}").append("\n")
        }
        builder.append("}").append("\n")
        builder.append("buildTypes:{").append("\n")
        for (def type : de.buildTypes) {
            builder.append("\t").append(type.name).append("{").append("\n")
            dumpDependable(builder, type)
            builder.append("\t").append("}").append("\n")
        }
        builder.append("}").append("\n")
        log(builder.toString())
    }

    protected static void log(String msg) {
        if (isDeBug)
            System.err.println("dependency plugin|>> [$msg]")
    }
}