package android.dev.decoupling

import org.gradle.TaskExecutionRequest
import org.gradle.api.NamedDomainObjectContainer
import org.gradle.api.Project

import java.util.concurrent.atomic.AtomicBoolean

class PluginV2Worker extends PluginWork {

    private final static String PROPERTY_DEBUG = "isDebug"

    private final static AtomicBoolean mIsDebug = new AtomicBoolean()

    @Override
    void apply(Project project) {
        if (!project.rootProject.hasProperty(PROPERTY_MAIN_MODULAR)) {
            throw new RuntimeException("you must set mainModular in root project gradle.properties")
        }

        if (project.rootProject.hasProperty(PROPERTY_DEBUG)) {
            mIsDebug.set((boolean) project.rootProject.property(PROPERTY_DEBUG))
        } else {
            mIsDebug.set(true)
        }
        String currentModule = project.name
        String mainModule = project.rootProject.property(PROPERTY_MAIN_MODULAR)
        String independentModule
        if (project.rootProject.hasProperty(PROPERTY_INDEPENDENT_MODULAR)) {
            independentModule = project.rootProject.property(PROPERTY_INDEPENDENT_MODULAR)
        } else {
            independentModule = mainModule
        }
        final boolean isAppApply = currentModule == mainModule || currentModule == independentModule
        List<TaskExecutionRequest> tasks = project.gradle.startParameter.getTaskRequests()
        dumpTaskInfos(currentModule, tasks)
        final ProjectTaskHolder holder = matchProjectTask(currentModule, tasks)
        final boolean isAssembleTask = holder == null ? false : holder.isAssembleTask()
        log("apply module $currentModule with [$mainModule,$independentModule] " + (isAssembleTask ? "be assembled" : "be build"))
        if (isAssembleTask && !isAppApply) {
            applyProjectLibrary(project)
        } else {
            applyProjectApplication(project, currentModule == mainModule)
        }
        NamedDomainObjectContainer<BuildTypes> buildTypes = project.container(BuildTypes.class)
        NamedDomainObjectContainer<Flavors> flavors = project.container(Flavors.class)
        DependencyExtension extension = new DependencyExtension(flavors, buildTypes)
        project.extensions.add(EXTENSION_NAME, extension)
        project.afterEvaluate {
            DependencyExtension de = project.extensions.getByName(EXTENSION_NAME)

            if (de == null) {
                return
            }
            if (mIsDebug.get()) {
                dumpUserConfiguration(de)
            }
            if (mIsDebug.get()) {
                log(holder.toString())
            }
            if (!isAssembleTask) {
                return
            }
            Closure sharedClosure = de.mSharedClosure
            if (sharedClosure != null) {
                sharedClosure.call()
            }
            final boolean isBatchTask = holder.isBatchTask()
            Flavors matchedFlavor
            if (isBatchTask || (matchedFlavor = matchFlavor(holder, de)) == null) {
                if (de.flavors != null && !de.flavors.isEmpty()) {
                    for (Flavors f : de.flavors) {
                        batchedCompile(project, f.name, f.mBeans)
                    }
                }
            } else {
                dependCompile(project, matchedFlavor.mBeans)
            }
            BuildTypes matchedBuildType
            if (isBatchTask || (matchedBuildType = matchBuildType(holder, de)) == null) {
                if (de.buildTypes != null && !de.buildTypes.isEmpty()) {
                    for (BuildTypes b : de.buildTypes) {
                        batchedCompile(project,b.name,b.mBeans)
                    }
                }
                return
            }
            dependCompile(project, matchedBuildType.mBeans)
        }
    }

    private
    static ProjectTaskHolder matchProjectTask(String currentModule, List<TaskExecutionRequest> requestList) {
        if (requestList == null || requestList.isEmpty()) {
            return null
        }
        ProjectTaskHolder holder = new ProjectTaskHolder(currentModule)
        for (TaskExecutionRequest request : requestList) {
            List<String> args = request.getArgs()
            for (String arg : args) {
                String assembleTask = parseRealTaskName(currentModule, arg)
                if (assembleTask == null) {
                    continue
                }
                if (assembleTask.contains("assemble") || assembleTask.contains("aR")
                        || assembleTask.contains("resguard")) {
                    holder.addTaskName(assembleTask)
                }
            }
        }
        return holder
    }

    private static String parseRealTaskName(String moduleName, String arg) {
        if (arg.contains(":")) {
            if (!arg.contains(moduleName)) {
                return null
            }
            String[] params = arg.split(":")
            return params[params.length - 1].toLowerCase()
        } else {
            return arg.toLowerCase()
        }
    }

    private static void dumpTaskInfos(String currentModule, List<TaskExecutionRequest> tasks) {
        if (mIsDebug.get()) {
            StringBuilder builder = new StringBuilder()
            for (TaskExecutionRequest request : tasks) {
                builder.append(currentModule).append("{").append("\n")
                for (String arg : request.getArgs()) {
                    builder.append(arg).append("\n")
                }
                builder.append("}").append("\n")
            }
            log(builder.toString())
        }
    }

    private static Flavors matchFlavor(ProjectTaskHolder holder, DependencyExtension de) {
        Flavors f
        for (String task : holder.tasks) {
            f = getFlavor(task, de)
            if (f != null) {
                return f
            }
        }
        return null
    }

    private static BuildTypes matchBuildType(ProjectTaskHolder holder, DependencyExtension de) {
        BuildTypes f
        for (String task : holder.tasks) {
            f = getBuildType(task, de)
            if (f != null) {
                return f
            }
        }
        return null
    }

    private
    static void batchedCompile(Project project, String modifyName, List<CompileCmd> cmdList) {
        if (cmdList != null && !cmdList.isEmpty()) {
            for (def cmd : cmdList) {
                def dep = cmd.dependency
                int type = dep.getType()
                def target = dep.getDepend(project)
                String configuration = formatConfiguration(modifyName, cmd.configuration)
                log("$configuration " + dep.toString() + " [$type]")
                project.dependencies.add(configuration, target)
            }
        }

    }

    private static String formatConfiguration(String modifyName, String configuration) {
        char c = configuration.charAt(0)
        if (c.isUpperCase()) {
            return modifyName + configuration
        }
        char[] config = configuration.toCharArray()
        config[0] = c.toUpperCase()
        return modifyName + (new String(config))
    }
}