package android.dev.decoupling

import org.gradle.api.NamedDomainObjectContainer
import org.gradle.api.Plugin
import org.gradle.api.Project

import java.util.concurrent.atomic.AtomicInteger

/**
 * Created by wbs on 2018/4/4 0004.
 */

class DependencyPlugin implements Plugin<Project> {

    private static boolean isDeBug = true

    private final static String EXTENSION_NAME = "dependencyOptions"

    private final static String PROPERTY_MAIN_MODULAR = "main_modular"

    private final static String PROPERTY_INDEPENDENT_MODULAR = "independent_modular"

    @Override
    void apply(Project project) {
        if (!project.rootProject.hasProperty(PROPERTY_MAIN_MODULAR)) {
            throw new RuntimeException("you must set mainModular in root project gradle.properties")
        }
        String currentModule = project.name
        String mainModule = project.rootProject.property(PROPERTY_MAIN_MODULAR)
        String independentModule
        if (project.rootProject.hasProperty(PROPERTY_INDEPENDENT_MODULAR)) {
            independentModule = project.rootProject.property(PROPERTY_INDEPENDENT_MODULAR)
        } else {
            independentModule = mainModule
        }

        List<String> taskNames = project.gradle.startParameter.taskNames

        final String task = findAssembleTaskRealName(taskNames)

        final boolean isAssembleTask = task != null
        final boolean isAppApply = currentModule == mainModule || currentModule == independentModule

        log("apply module $currentModule with [$mainModule,$independentModule] " + (isAssembleTask ? "be assembled" : "be build"))

        if (isAssembleTask && !isAppApply) {
            project.apply plugin: 'com.android.library'
        } else {
            project.apply plugin: 'com.android.application'
            if (currentModule != mainModule) {
                project.android.sourceSets {
                    main {
                        manifest.srcFile 'src/main/independent/AndroidManifest.xml'
                        java.srcDirs += ['src/main/independent/java']
                        res.srcDirs += ['src/main/independent/res']
                    }
                }
            }
        }

        NamedDomainObjectContainer<BuildTypes> buildTypes = project.container(BuildTypes.class)
        NamedDomainObjectContainer<Flavors> flavors = project.container(Flavors.class)
        DependencyExtension extension = new DependencyExtension(flavors, buildTypes)
        project.extensions.add(EXTENSION_NAME, extension)
        project.afterEvaluate {
            DependencyExtension de = project.extensions.getByName(EXTENSION_NAME)

            if (de == null) {
                return
            }
            isDeBug = de.isDebug
            if (isDeBug) {
                StringBuilder builder = new StringBuilder()
                builder.append("user configuration list:").append("\n")
                builder.append("flavors:{").append("\n")
                for (def flavor : de.flavors) {
                    builder.append("\t").append(flavor.name).append("{").append("\n")
                    dumpDependable(builder, flavor)
                    builder.append("\t").append("}").append("\n")
                }
                builder.append("}").append("\n")
                builder.append("buildTypes:{").append("\n")
                for (def type : de.buildTypes) {
                    builder.append("\t").append(type.name).append("{").append("\n")
                    dumpDependable(builder, type)
                    builder.append("\t").append("}").append("\n")
                }
                builder.append("}").append("\n")
                log(builder.toString())
            }
            if (!isAssembleTask) {
                return
            }

            Closure sharedClosure = de.mSharedClosure
            if (sharedClosure != null) {
                sharedClosure.call()
            }

            Flavors flavor = getFlavor(task, de)
            BuildTypes buildType = getBuildType(task, de)
            String typeName = buildType==null?"None":buildType.name
            String flavorName = flavor==null?"None":flavor.name
            log("matched buildType[$typeName],flavor[$flavorName]")
            if(flavor!=null){
                dependCompile(project, flavor.mBeans)
            }
            if(buildType!=null){
                dependCompile(project, buildType.mBeans)
            }

            log("apply plugin android.module.depends end")
        }
    }

    private static void dumpDependable(StringBuilder builder, CompileBean dependable) {
        List<CompileCmd> dep = dependable.mBeans
        for (def cmd : dep) {
            def obj = cmd.dependency
            builder.append("\t\t").append(cmd.configuration).append(" ").append(obj.toString()).append("\n")
        }
    }

    private static Flavors getFlavor(String taskName, DependencyExtension extension) {
        NamedDomainObjectContainer<Flavors> flavors = extension.flavors
        if (flavors == null || flavors.isEmpty()) {
            return null
        }
        for (def flavor : flavors) {
            if (taskName.toLowerCase().contains(flavor.name.toLowerCase())) {
                return flavor
            }
        }
        return null
    }

    private static BuildTypes getBuildType(String taskName, DependencyExtension extension) {
        NamedDomainObjectContainer<BuildTypes> buildTypes = extension.buildTypes
        if (buildTypes == null || buildTypes.isEmpty()) {
            return null
        }
        for (def buildType : buildTypes) {
            if (taskName.toLowerCase().contains(buildType.name.toLowerCase())) {
                return buildType
            }
        }
        return null
    }

    private static void dependCompile(Project project, List<CompileCmd> cmdList) {
        if (cmdList != null && !cmdList.isEmpty()) {
            for (def cmd : cmdList) {
                def dep = cmd.dependency
                int type = dep.getType()
                log("$cmd.configuration " + dep.toString() + " [$type]")
                def target = dep.getDepend(project)
                project.dependencies.add(cmd.configuration, target)
            }
        }

    }

    private static String findAssembleTaskRealName(List<String> names) {
        for (String task : names) {
            String lowercaseName = task.toLowerCase()
            if (lowercaseName.contains("assemble") || task.contains("aR")
                    || lowercaseName.contains("resguard")) {
                return task
            }
        }
        return null
    }

    private static void log(String msg) {
        if (isDeBug)
            System.out.println("dependency plugin|>>>>>>>>>>>>>>>>>>> [$msg]")
    }

}
