
package android.dev.base.life;

import android.arch.lifecycle.Lifecycle.Event;
import android.dev.base.life.transformer.BindLifecycleCompletableTransformer;
import android.dev.base.life.transformer.BindLifecycleFlowableTransformer;
import android.dev.base.life.transformer.BindLifecycleMaybeTransformer;
import android.dev.base.life.transformer.BindLifecycleObservableTransformer;
import android.dev.base.life.transformer.BindLifecycleSingleTransformer;
import android.support.annotation.NonNull;

import io.reactivex.CompletableTransformer;
import io.reactivex.Flowable;
import io.reactivex.FlowableTransformer;
import io.reactivex.MaybeTransformer;
import io.reactivex.Observable;
import io.reactivex.ObservableSource;
import io.reactivex.ObservableTransformer;
import io.reactivex.SingleTransformer;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.schedulers.Schedulers;

public class RxLifecycle {
    private final LifecyclePublisher lifecyclePublisher;

    private RxLifecycle() throws IllegalAccessException {
        throw new IllegalAccessException();
    }

    private RxLifecycle(@NonNull LifecyclePublisher lifecyclePublisher) {
        this.lifecyclePublisher = lifecyclePublisher;
    }

    public static RxLifecycle bind(LifecycleComponent component){
        return new RxLifecycle(component.getLifeCyclePublisher());
    }

    public Flowable<Event> asFlowable() {
        return lifecyclePublisher.getBehavior();
    }

    public Observable<Event> asObservable() {
        return lifecyclePublisher.getBehavior().toObservable();
    }

    public <T> FlowableTransformer<T, T> withFlowable() {
        return new BindLifecycleFlowableTransformer<T>(lifecyclePublisher.getBehavior());
    }

    public <T> ObservableTransformer<T, T> withObservable() {
        return new BindLifecycleObservableTransformer<T>(lifecyclePublisher.getBehavior());
    }

    public CompletableTransformer withCompletable() {
        return new BindLifecycleCompletableTransformer(lifecyclePublisher.getBehavior());
    }

    public <T> SingleTransformer<T, T> withSingle() {
        return new BindLifecycleSingleTransformer<T>(lifecyclePublisher.getBehavior());
    }

    public <T> MaybeTransformer<T, T> withMaybe() {
        return new BindLifecycleMaybeTransformer<T>(lifecyclePublisher.getBehavior());
    }
    //ui dispatch
    static class RxTransformer<T> implements ObservableTransformer<T, T> {
        @Override
        public ObservableSource<T> apply(@io.reactivex.annotations.NonNull Observable<T> upstream) {
            return upstream
                    .subscribeOn(Schedulers.io())
                    .observeOn(AndroidSchedulers.mainThread());
        }
    }

    private final static ObservableTransformer SCHEDULE_TRANSFORMER = new RxTransformer();

    @SuppressWarnings("unchecked")
    public static <T> ObservableTransformer<T, T> callerSchedulers() {
        return SCHEDULE_TRANSFORMER;
    }
}
