package android.dev.extension.graphic;

import android.content.res.ColorStateList;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.GradientDrawable;
import android.graphics.drawable.StateListDrawable;
import android.support.v4.view.ViewCompat;
import android.view.View;
import android.widget.TextView;

/**
 * Created by oeager on 2016/11/15.
 */

public class DrawableHelper {


    public static Builder of(View view) {
        return new Builder(view);
    }

    public static TextBuilder of(TextView textView){
        return new TextBuilder(textView);
    }

    public static class Builder {

        private GradientDrawable mNormalBackground;

        private GradientDrawable mPressedBackground;

        private GradientDrawable mUnableBackground;

        private final StateListDrawable mDrawable;

        final int[][] mStates;

        private int mStrokeDashWidth;

        private int mStrokeDashGap;

        private boolean mRound = false;

         View mView;

        private Builder(View view) {
            mView = view;
            Drawable drawable = view.getBackground();
            if (drawable != null && drawable instanceof StateListDrawable) {
                mDrawable = (StateListDrawable) drawable;
            } else {
                mDrawable = new StateListDrawable();
            }
            mNormalBackground = new GradientDrawable();
            mPressedBackground = new GradientDrawable();
            mUnableBackground = new GradientDrawable();
            mStates = new int[4][];
            //pressed, focused, normal, unable
            mStates[0] = new int[]{android.R.attr.state_enabled, android.R.attr.state_pressed};
            mStates[1] = new int[]{android.R.attr.state_enabled, android.R.attr.state_focused};
            mStates[2] = new int[]{android.R.attr.state_enabled};
            mStates[3] = new int[]{-android.R.attr.state_enabled};


        }

        public Builder duration(int duration) {
            mDrawable.setEnterFadeDuration(duration);
            return this;
        }

        public Builder normalColor(int color) {
            mNormalBackground.setColor(color);
            return this;
        }

        public Builder pressedColor(int color) {
            mPressedBackground.setColor(color);
            return this;
        }

        public Builder unableColor(int color) {
            mUnableBackground.setColor(color);
            return this;
        }

        public Builder radius(float radius) {
            setRadius(radius);
            return this;
        }

        public Builder round(boolean round) {
            mRound = round;
            mView.addOnLayoutChangeListener(mListener);
            return this;
        }

        public Builder strokeDashWidth(int width) {
            mStrokeDashWidth = width;
            return this;
        }

        public Builder strokeDashGap(int size) {
            mStrokeDashGap = size;
            return this;
        }

        public Builder normalStroke(int width, int color) {
            mNormalBackground.setStroke(width, color, mStrokeDashWidth, mStrokeDashGap);
            return this;
        }

        public Builder pressedStroke(int width, int color) {
            mPressedBackground.setStroke(width, color, mStrokeDashWidth, mStrokeDashGap);
            return this;
        }

        public Builder unableStroke(int width, int color) {
            mUnableBackground.setStroke(width, color, mStrokeDashWidth, mStrokeDashGap);
            return this;
        }

        public Builder state(int[] states, Drawable drawable) {
            mDrawable.addState(states, drawable);
            return this;
        }

        public Builder build() {
            //pressed, focused, normal, unable
            mDrawable.addState(mStates[0], mPressedBackground);
            mDrawable.addState(mStates[1], mPressedBackground);
            mDrawable.addState(mStates[2], mNormalBackground);
            mDrawable.addState(mStates[3], mUnableBackground);
            return this;
        }


        public StateListDrawable get() {
            return mDrawable;
        }

        public void attach() {
            ViewCompat.setBackground(mView, mDrawable);
        }

        void setRadius(float radius) {
            mNormalBackground.setCornerRadius(radius);
            mPressedBackground.setCornerRadius(radius);
            mUnableBackground.setCornerRadius(radius);
        }

        private final View.OnLayoutChangeListener mListener = new View.OnLayoutChangeListener() {
            @Override
            public void onLayoutChange(View v, int left, int top, int right, int bottom, int oldLeft, int oldTop, int oldRight, int oldBottom) {
                int height = v.getMeasuredHeight();
                if (mRound) {
                    setRadius(height / 2f);
                }
                v.removeOnLayoutChangeListener(this);
            }
        };
    }

    public static class TextBuilder extends Builder {

        private ColorStateList mTextColorStateList;

        private int[] colors = new int[4];

        private TextBuilder(TextView view) {
            super(view);
            mTextColorStateList = view.getTextColors();
            int mDefaultNormalTextColor = mTextColorStateList.getColorForState(mStates[2], view.getCurrentTextColor());
            int mDefaultPressedTextColor = mTextColorStateList.getColorForState(mStates[0], view.getCurrentTextColor());
            int mDefaultUnableTextColor = mTextColorStateList.getColorForState(mStates[3], view.getCurrentTextColor());
            colors[0] = mDefaultPressedTextColor;
            colors[1] = mDefaultPressedTextColor;
            colors[2] = mDefaultNormalTextColor;
            colors[3] = mDefaultUnableTextColor;
        }

        public TextBuilder normalTextColor(int color) {
            colors[2] = color;
            setTextColor();
            return this;
        }

        public TextBuilder pressedTextColor(int color) {
            colors[0] = color;
            colors[1] = color;
            setTextColor();
            return this;
        }

        public TextBuilder unableTextColor(int color) {
            colors[3] = color;
            setTextColor();
            return this;
        }

        @Override
        public void attach() {
            super.attach();
            TextView textView = (TextView) mView;
            textView.setTextColor(mTextColorStateList);
        }

        public ColorStateList getTextColor(){
            return mTextColorStateList;
        }

        void setTextColor() {
            mTextColorStateList = new ColorStateList(mStates, colors);
        }
    }
}
