package android.dev.router;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.support.annotation.IntDef;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentActivity;
import android.support.v4.util.ArraySet;
import android.text.TextUtils;
import android.view.Window;

import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static android.dev.router.NamingPool.getBridgeQualifiedName;
import static android.dev.router.NamingPool.getLinkQueryInjectorName;
import static android.dev.router.NamingPool.getTransformedHostName;

/**
 * Created by wbs on 2017/11/28 0028.
 * We use URI to map each page in Android:
 * schema://host:port/path/?queryKey = queryValue#fragment
 * [schema] filter intent,Only the declarative schema can be responded to.
 * [host] The modular name,it will used for load routerTable
 * [port] none use
 * [path] The Activity used to point to the jump.
 * [query] The parameters will be send.
 * [fragment] just used for Activity contains Tab,for the Tab index;
 */

public final class Router {


    private static class Holder {
        private final static Router IMPL = new Router();
    }

    public static Router getDefault() {
        return Holder.IMPL;
    }

    private Router() {

    }

    public final static int TYPE_ACTIVITY = 0;

    public final static int TYPE_FRAGMENT = 1;

    public final static int TYPE_FRAGMENT_V4 = 2;

    final static String KEY_LINK_TYPE = "router_link_type";

    final static String KEY_FRAGMENT = "router_fragment";

    final static String KEY_PAGE = "router_page";


    @IntDef({TYPE_ACTIVITY, TYPE_FRAGMENT, TYPE_FRAGMENT_V4})
    public @interface LinkType {

    }

    public final static int INVALIDATE_REQUEST_CODE = Integer.MIN_VALUE;

    private final ArraySet<String> mSchemas = new ArraySet<>();

    private final List<RouterInterceptor> mInterceptors = new ArrayList<>();

    private final Map<String, List<ForwardTransformer>> mTransformers = new HashMap<>();

    private final ForwardTransformer mDefaultForwardTransformer = new DefaultForwardTransformer();

    private final AddressAuthenticator mAuthenticator = new AddressAuthenticator();

    private String mDefaultSchema;

    private String getPackageDefaultSchema(Context context) {
        if (mSchemas.isEmpty()) {
            return "";
        }
        String pkg = context.getPackageName();
        String[] pots = pkg.split(".");
        if (pots.length > 0) {
            for (String pot : pots) {
                if (mSchemas.contains(pot)) {
                    return pot;
                }
            }
        }
        return mSchemas.valueAt(0);
    }

    public String getDefaultSchema() {
        if (mDefaultSchema != null) {
            return mDefaultSchema;
        }
        if (!mSchemas.isEmpty()) {
            mDefaultSchema = mSchemas.valueAt(0);
            return mDefaultSchema;
        }
        return "";
    }

    private GlobalRelegationHandler mGlobalHandler;

    private DirectionalRelegationHandler mRelegationHandler;

    public void registerSchema(String schema) {
        if (schema == null) {
            return;
        }
        mSchemas.add(schema);
    }

    public void enableDefaultSchema(Context context) {
        mDefaultSchema = getPackageDefaultSchema(context);
    }

    public void enableDefaultSchema(String schema) {
        if (schema == null) {
            return;
        }
        if (!mSchemas.contains(schema)) {
            registerSchema(schema);
        }
        this.mDefaultSchema = schema;
    }

    public void unregisterSchema(String schema) {
        if (schema == null) {
            return;
        }
        mSchemas.remove(schema);
    }

    public boolean isSchemaValid(String schema) {
        if (schema == null)
            return false;
        if (mSchemas.isEmpty())
            return true;
        return mSchemas.contains(schema);
    }

    public void registerGlobalRelegationHandler(GlobalRelegationHandler handler) {
        this.mGlobalHandler = handler;
    }

    public void registerDirectionalRelegationHandler(DirectionalRelegationHandler handler) {
        this.mRelegationHandler = handler;
    }

    public void registerRouterBridge(Bridge bridge) {
        mAuthenticator.registerBridge(bridge);
    }

    public void registerRouterBridge(String hostName) {
        try {
            Class<?> _cls = Class.forName(getBridgeQualifiedName(hostName));
            Object clazz = _cls.newInstance();
            if (clazz instanceof Bridge) {
                registerRouterBridge((Bridge) clazz);
            }
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        } catch (InstantiationException e) {
            e.printStackTrace();
        }
    }

    public void addRouterInterceptor(RouterInterceptor interceptor) {
        mInterceptors.add(interceptor);
    }

    public void registerForwardTransformer(Class<? extends Activity> _Cls, ForwardTransformer transformer) {
        if (transformer == null) {
            return;
        }
        synchronized (mTransformers) {
            String clsName = _Cls.getName();
            List<ForwardTransformer> transformers = mTransformers.get(clsName);
            if (transformers == null) {
                transformers = new ArrayList<>();
                mTransformers.put(clsName, transformers);
            }
            transformers.add(transformer);
        }

    }

    public void unregisterForwardTransformer(Class<? extends Activity> _Target, Class<? extends ForwardTransformer> _Cls) {
        synchronized (mTransformers) {
            String clsName = _Target.getName();
            List<ForwardTransformer> transformers = mTransformers.get(clsName);
            if (transformers != null) {
                Iterator<ForwardTransformer> transformerIterator = transformers.iterator();
                while (transformerIterator.hasNext()) {
                    ForwardTransformer ft = transformerIterator.next();
                    if (ft.getClass().equals(_Cls)) {
                        transformerIterator.remove();
                        break;
                    }
                }
            }
        }
    }

    public void unregisterForwardTransformer(Class<? extends Activity> _Cls) {
        synchronized (mTransformers) {
            mTransformers.remove(_Cls.getName());
        }
    }

    ForwardTransformer getTargetForwardTransformer(Context mContext, String mTarget) {
        synchronized (mTransformers) {
            List<ForwardTransformer> transformers = mTransformers.get(mTarget);
            if (transformers != null) {
                for (ForwardTransformer forwardTransformer : transformers) {
                    if (forwardTransformer.fit(mContext)) {
                        return forwardTransformer;
                    }
                }
            }
        }
        return mDefaultForwardTransformer;
    }

    public void addUriTransformer(UriTransformer transformer) {
        mAuthenticator.addUriTransformer(transformer);
    }


    public void enter(Context context, Uri uri) {
        enter(context, uri, null);
    }

    void enter(Context context, Uri uri, Bundle extraData) {
        enter(context, uri, extraData, INVALIDATE_REQUEST_CODE);
    }

    public void enter(Context context, String url, int requestCode) {
        Uri uri;
        try {
            uri = Uri.parse(url);
        } catch (Exception e) {
            globalRelegation(url, ErrorType.INVALIDATE_URI);
            return;
        }
        enter(context, uri, requestCode);
    }

    public void enter(Context context, String url) {
        enter(context, url, INVALIDATE_REQUEST_CODE);
    }

    public void enter(Context context, Uri uri, int requestCode) {
        enter(context, uri, null, requestCode);
    }

    void enter(Context context, Uri uri, Bundle extraData, int requestCode) {
        if (uri == null) {
            globalRelegation(null, ErrorType.INVALIDATE_URI);
            return;
        }
        try {
            uri = mAuthenticator.performTransformUri(context, uri);
        } catch (Exception e) {
            globalRelegation(null, ErrorType.INVALIDATE_URI);
            return;
        }

        String schema = uri.getScheme();
        if (!isSchemaValid(schema)) {
            globalRelegation(uri.toString(), ErrorType.INVALIDATE_SCHEMA);
            return;
        }
        Address address = mAuthenticator.getAddress(uri);
        if (address == null) {
            globalRelegation(uri.toString(), ErrorType.IP_NOT_FOUND);
            return;
        }
        Set<String> names = uri.getQueryParameterNames();
        String page = uri.getFragment();
        if (names != null) {
            if (extraData == null) {
                extraData = new Bundle();
            }
            for (String name : names) {
                extraData.putString(name, uri.getQueryParameter(name));
            }
        }
        if(page!=null){
            if(extraData ==null){
                extraData = new Bundle();
            }
            extraData.putString(KEY_PAGE,page);
        }

        enter(context, address, extraData, requestCode);
    }

    private void enter(Context context, Address address, Bundle parameters, int requestCode) {
        List<RouterInterceptor> interceptors = new ArrayList<>(mInterceptors.size() + 1);
        interceptors.addAll(mInterceptors);
        interceptors.add(new ForwardRouterInterceptor(context));
        Chain chain = new Chain(interceptors, 0);
        try {
            Channel channel = chain.proceed(address, parameters, requestCode);
            if (channel == null) {
                directionalRelegation(address, parameters);
                return;
            }
            if (channel.mStatus == Channel.STATUS_OK) {
                channel.enter(this, requestCode);
            } else if (channel.mStatus == Channel.STATUS_ASYNC) {
                //todo nothing
            }

        } catch (Exception error) {
            directionalRelegation(address, parameters);
        }

    }

    public static PathLinker.AppLink link(String path) {
        return PathLinker.getAppLink(path);
    }

    @LinkType
    public int queryLinkType(Address address) {
        if (address == null) {
            throw new IllegalArgumentException();
        }
        return address.mType;
    }

    @LinkType
    public int queryLinkType(Activity activity) {
        Intent intent = activity.getIntent();
        return intent.getIntExtra(KEY_LINK_TYPE, TYPE_ACTIVITY);
    }

    @SuppressWarnings("unchecked")
    public void attachFragmentActivity(Activity activity, int containerId) {
        int type = queryLinkType(activity);
        if (type == TYPE_ACTIVITY) {
            return;
        }
        Intent intent = activity.getIntent();
        String fragmentName = intent.getStringExtra(KEY_FRAGMENT);
        if (TextUtils.isEmpty(fragmentName)) {
            return;
        }
        Bundle bundle = intent.getExtras();
        if (bundle != null) {
            bundle.remove(KEY_FRAGMENT);
            bundle.remove(KEY_LINK_TYPE);
        }
        if (containerId == 0) {
            containerId = Window.ID_ANDROID_CONTENT;
        }
        if (type == TYPE_FRAGMENT_V4 && activity instanceof FragmentActivity) {
            Fragment fragment = Fragment.instantiate(activity, fragmentName, bundle);
            ((FragmentActivity) activity).getSupportFragmentManager().beginTransaction()
                    .add(containerId, fragment, fragmentName)
                    .commitAllowingStateLoss();
        } else {
            android.app.Fragment fragment = android.app.Fragment.instantiate(activity, fragmentName, bundle);
            activity.getFragmentManager().beginTransaction()
                    .add(containerId, fragment, fragmentName)
                    .commitAllowingStateLoss();
        }

    }

    public String getRouterPage(Activity activity){
        Intent intent = activity.getIntent();
        return intent.getStringExtra(KEY_PAGE);
    }

    public int getRouterPageIndex(Activity activity){
        String page = getRouterPage(activity);
        if(page==null){
            return -1;
        }
        return QueryParser.parseInt(page);
    }


    public void injectQueryData(Object target) {
        String injectClazzName = getLinkQueryInjectorName(target);
        Context context;
        if (target instanceof Activity) {
            context = (Context) target;
        } else if (target instanceof android.app.Fragment) {
            context = ((android.app.Fragment) target).getActivity();
        } else if (target instanceof Fragment) {
            context = ((Fragment) target).getActivity();
        } else {
            throw new RuntimeException("queryData just support get in fragment or activity");
        }
        LinkQueryInjector injector = QueryBinder.findLinkInjector(context, injectClazzName);
        if (injector != null) {
            injector.inject(target);
        }
    }

    private void globalRelegation(String what, ErrorType why) {
        if (mGlobalHandler != null) {
            mGlobalHandler.handleRelegation(mAuthenticator, what, why);
        }
    }

    private void directionalRelegation(Address where, Bundle what) {
        if (mRelegationHandler != null) {
            mRelegationHandler.handleRelegation(where, what);
        }
    }

}
