package android.dev.base.utils;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.location.LocationManager;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.net.wifi.WifiInfo;
import android.net.wifi.WifiManager;
import android.support.annotation.RequiresPermission;
import android.support.v4.content.ContextCompat;
import android.telephony.TelephonyManager;

/**
 * 跟网络相关的工具类
 */
public class Connectivity {
    private Connectivity() {
        /* cannot be instantiated */
        throw new UnsupportedOperationException("cannot be instantiated");
    }

    static final String NETWORK_TYPE_LTE = "LTE"; // 4G
    static final String NETWORK_TYPE_WIFI = "WIFI";// WIFI
    static final String NETWORK_TYPE_GPRS = "GPRS"; // 2G
    static final String NETWORK_TYPE_IDEN = "IDEN"; // 2G
    static final String NETWORK_TYPE_EDGE = "EDGE"; // 2.5G
    static final String NETWORK_TYPE_UMTS = "UMTS"; // 3G
    static final String NETWORK_TYPE_EVDO_0 = "EVDO_0";// 3G
    static final String NETWORK_TYPE_HSPA = "HSPA"; // 3G->4G
    static final String NETWORK_TYPE_HSUPA = "HSUPA"; // 3G->4G
    static final String NETWORK_TYPE_HSDPA = "HSDPA"; // 3G->4G
    static final String NETWORK_TYPE_EVDO_A = "EVDO_A";// 3G->4G
    static final String NETWORK_TYPE_1xRTT = "1xRTT"; // CDMA2000 1xRTT
    static final String NETWORK_TYPE_CDMA = "CDMA"; // CDMAONE 2G
    static final String NETWORK_TYPE_HSPAP = "HSPAP";
    static final String NETWORK_TYPE_EHRPD = "EHRPD";
    static final String NETWORK_TYPE_EVDO_B = "EVDO_B";
    static final String NETWORK_TYPE_UNKNOWN = "UNKNOWN";
    static final String NETWORK_TYPE_NONE = "NONE";

    /**
     * 描述：判断网络是否有效.
     *
     * @param context the context
     * @return true, if is network available
     */
    @RequiresPermission(android.Manifest.permission.ACCESS_NETWORK_STATE)
    public static boolean isNetworkAvailable(Context context) {
        try {
            ConnectivityManager connectivity = (ConnectivityManager) context
                    .getSystemService(Context.CONNECTIVITY_SERVICE);
            if (connectivity != null) {
                NetworkInfo info = connectivity.getActiveNetworkInfo();
                if (info != null && info.isConnected()) {
                    if (info.getState() == NetworkInfo.State.CONNECTED) {
                        return true;
                    }
                }
            }
        } catch (Exception e) {
            return false;
        }
        return false;
    }

    /**
     * Gps是否打开
     * 需要android.permission.ACCESS_FINE_LOCATION权限
     *
     * @param context the context
     * @return true, if is gps enabled
     */
    @RequiresPermission(Manifest.permission.ACCESS_FINE_LOCATION)
    public static boolean isGpsEnabled(Context context) {
        LocationManager lm = (LocationManager) context.getSystemService(Context.LOCATION_SERVICE);
        return lm.isProviderEnabled(LocationManager.GPS_PROVIDER);
    }

    /**
     * wifi是否打开.
     *
     * @param context the context
     * @return true, if is wifi enabled
     */
    @RequiresPermission(android.Manifest.permission.ACCESS_NETWORK_STATE)
    public static boolean isWifiEnabled(Context context) {
        ConnectivityManager mgrConn = (ConnectivityManager) context
                .getSystemService(Context.CONNECTIVITY_SERVICE);
        TelephonyManager mgrTel = (TelephonyManager) context
                .getSystemService(Context.TELEPHONY_SERVICE);
        if (mgrConn != null && ((mgrConn.getActiveNetworkInfo() != null
                && mgrConn.getActiveNetworkInfo().getState() == NetworkInfo.State.CONNECTED))) {
            return true;
        }
        if (mgrTel != null) {
            if (mgrTel.getNetworkType() == TelephonyManager.NETWORK_TYPE_UMTS) {
                return true;
            }
        }
        return false;
    }

    /**
     * 判断当前网络是否是wifi网络.
     *
     * @param context the context
     * @return boolean
     */
    @RequiresPermission(android.Manifest.permission.ACCESS_NETWORK_STATE)
    public static boolean isWifi(Context context) {
        ConnectivityManager connectivityManager = (ConnectivityManager) context
                .getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo activeNetInfo = null;
        if (connectivityManager != null) {
            activeNetInfo = connectivityManager.getActiveNetworkInfo();
        }
        return activeNetInfo != null
                && activeNetInfo.getType() == ConnectivityManager.TYPE_WIFI;
    }

    /**
     * 判断当前网络是否是3G网络.
     *
     * @param context the context
     * @return boolean
     * @deprecated see {@link #isMobile(Context)}
     */
    @RequiresPermission(android.Manifest.permission.ACCESS_NETWORK_STATE)
    public static boolean is3G(Context context) {
        return isMobile(context);
    }

    @RequiresPermission(android.Manifest.permission.ACCESS_NETWORK_STATE)
    public static boolean isMobile(Context context) {
        ConnectivityManager connectivityManager = (ConnectivityManager) context
                .getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo activeNetInfo = connectivityManager.getActiveNetworkInfo();
        return activeNetInfo != null
                && activeNetInfo.getType() == ConnectivityManager.TYPE_MOBILE;
    }

    @RequiresPermission(android.Manifest.permission.ACCESS_NETWORK_STATE)
    public static boolean isAvailableMobile(Context context) {
        ConnectivityManager connectivityManager = (ConnectivityManager) context
                .getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo activeNetInfo = connectivityManager.getActiveNetworkInfo();
        return activeNetInfo != null
                && activeNetInfo.getType() == ConnectivityManager.TYPE_MOBILE && activeNetInfo.isAvailable();
    }

    @RequiresPermission(android.Manifest.permission.ACCESS_NETWORK_STATE)
    public static int getNetworkType(Context ctx) {
        ConnectivityManager cm = (ConnectivityManager) ctx.getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo ni = cm.getActiveNetworkInfo();
        if (ni == null || !ni.isConnected()) {
            return -1;
        }
        return ni.getType();
    }

    public static void openNetworkSetting(Context ctx) {
        try {
            Intent intent;
            intent = new Intent(android.provider.Settings.ACTION_WIRELESS_SETTINGS);
            ctx.startActivity(intent);
        } catch (Exception e) {
            try {
                openNetworkSetting2((Activity) ctx);
            } catch (Exception ignored) {

            }
        }
    }

    /**
     * 打开网络设置界面
     */
    public static void openNetworkSetting2(Activity activity) {
        Intent intent = new Intent("/");
        ComponentName cm = new ComponentName("com.android.settings",
                "com.android.settings.WirelessSettings");
        intent.setComponent(cm);
        intent.setAction("android.intent.action.VIEW");
        activity.startActivityForResult(intent, 0);
    }

    public static boolean tryOpenWifiSettings(Context context) {
        boolean succeed = openWifiSettings(context);
        if (!succeed) {
            succeed = openWifiSettings2(context);
        }
        if (!succeed) {
            succeed = openWifiSettings3(context);
        }
        return succeed;
    }

    public static boolean openWifiSettings(Context context) {
        try {
            Intent intent = new Intent();
            intent.setAction("android.net.wifi.PICK_WIFI_NETWORK");
            context.startActivity(intent);
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    public static boolean openWifiSettings2(Context context) {
        try {
            context.startActivity(new Intent(android.provider.Settings.ACTION_WIFI_SETTINGS));
            return true;
        } catch (Exception e) {
            return false;
        }

    }

    public static boolean openWifiSettings3(Context context) {
        try {
            Intent i = new Intent();
            i.setClassName("com.android.settings", "com.android.settings.Settings$WifiSettingsActivity");
            context.startActivity(i);
            return true;
        } catch (Exception e) {
            return false;
        }

    }

    @SuppressLint("HardwareIds")
    @RequiresPermission(allOf = {Manifest.permission.CHANGE_WIFI_STATE, Manifest.permission.ACCESS_WIFI_STATE})
    public static String getMacAddress(Context mContext) {
        WifiManager wifiManager = (WifiManager) mContext.getApplicationContext()
                .getSystemService(Context.WIFI_SERVICE);
        // 判断wifi是否开启
        if (!wifiManager.isWifiEnabled()) {
            if (ContextCompat.checkSelfPermission(mContext, android.Manifest.permission.CHANGE_WIFI_STATE) == PackageManager.PERMISSION_GRANTED) {
                wifiManager.setWifiEnabled(true);
            } else {
                return "";
            }

        }
        if (ContextCompat.checkSelfPermission(mContext, android.Manifest.permission.ACCESS_WIFI_STATE) == PackageManager.PERMISSION_GRANTED) {
            WifiInfo wifiInfo = wifiManager.getConnectionInfo();
            return wifiInfo.getMacAddress();
        }
        return "";
    }

    @RequiresPermission(allOf = {Manifest.permission.CHANGE_WIFI_STATE, Manifest.permission.ACCESS_WIFI_STATE})
    public static String getPhoneIp(Context mContext) {
        WifiManager wifiManager = (WifiManager) mContext.getApplicationContext()
                .getSystemService(Context.WIFI_SERVICE);
        // 判断wifi是否开启
        if (wifiManager != null && !wifiManager.isWifiEnabled()) {
            if (ContextCompat.checkSelfPermission(mContext, Manifest.permission.CHANGE_WIFI_STATE) == PackageManager.PERMISSION_GRANTED) {
                wifiManager.setWifiEnabled(true);
            } else {
                return "";
            }
        }
        if (ContextCompat.checkSelfPermission(mContext, android.Manifest.permission.ACCESS_WIFI_STATE) == PackageManager.PERMISSION_GRANTED) {
            WifiInfo wifiInfo = null;
            if (wifiManager != null) {
                wifiInfo = wifiManager.getConnectionInfo();
            }
            int ipAddress = 0;
            if (wifiInfo != null) {
                ipAddress = wifiInfo.getIpAddress();
            }
            return intToIp(ipAddress);
        }
        return "";
    }

    private static String intToIp(int i) {

        return (i & 0xFF) + "." + ((i >> 8) & 0xFF) + "." + ((i >> 16) & 0xFF)
                + "." + (i >> 24 & 0xFF);
    }

    /**
     * 网络类型（文字形式）
     */
    @RequiresPermission(android.Manifest.permission.ACCESS_NETWORK_STATE)
    public static String getNetTypeString(Context mContext) {
        ConnectivityManager mConnectivityManager = (ConnectivityManager) mContext
                .getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo info = null;
        if (mConnectivityManager != null) {
            info = mConnectivityManager.getActiveNetworkInfo();
        }
        if (info != null && info.isAvailable()) {
            if (info.getType() == ConnectivityManager.TYPE_WIFI) {
                return NETWORK_TYPE_WIFI;
            }
            int type = info.getSubtype();
            switch (type) {
                case TelephonyManager.NETWORK_TYPE_1xRTT:
                    return NETWORK_TYPE_1xRTT;
                case TelephonyManager.NETWORK_TYPE_CDMA:
                    return NETWORK_TYPE_CDMA;
                case TelephonyManager.NETWORK_TYPE_EDGE:
                    return NETWORK_TYPE_EDGE;
                case TelephonyManager.NETWORK_TYPE_EVDO_0:
                    return NETWORK_TYPE_EVDO_0;
                case TelephonyManager.NETWORK_TYPE_EVDO_A:
                    return NETWORK_TYPE_EVDO_A;
                case TelephonyManager.NETWORK_TYPE_GPRS:
                    return NETWORK_TYPE_GPRS;
                case TelephonyManager.NETWORK_TYPE_HSDPA:
                    return NETWORK_TYPE_HSDPA;
                case TelephonyManager.NETWORK_TYPE_HSPA:
                    return NETWORK_TYPE_HSPA;
                case TelephonyManager.NETWORK_TYPE_HSUPA:
                    return NETWORK_TYPE_HSUPA;
                case TelephonyManager.NETWORK_TYPE_IDEN:
                    return NETWORK_TYPE_IDEN;
                case TelephonyManager.NETWORK_TYPE_UMTS:
                    return NETWORK_TYPE_UMTS;
                case TelephonyManager.NETWORK_TYPE_HSPAP:
                    return NETWORK_TYPE_HSPAP;
                case TelephonyManager.NETWORK_TYPE_EHRPD:
                    return NETWORK_TYPE_EHRPD;
                case TelephonyManager.NETWORK_TYPE_LTE:
                    return NETWORK_TYPE_LTE;
                case TelephonyManager.NETWORK_TYPE_EVDO_B:
                    return NETWORK_TYPE_EVDO_B;
                case TelephonyManager.NETWORK_TYPE_UNKNOWN:
                    return NETWORK_TYPE_UNKNOWN;
                default:
                    return NETWORK_TYPE_UNKNOWN;
            }
        } else {
            return NETWORK_TYPE_NONE;
        }
    }

}
