package android.dev.base.arch;

import android.app.Application;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by wbs on 2018/3/27 0027.
 */

public final class ModuleLoader implements Pool<Module> {

    private ModuleLoader() {
    }

    private final static ModuleLoader _IMPL = new ModuleLoader();

    public static ModuleLoader getDefault() {
        return _IMPL;
    }

    private final Map<String, Module> mModules = new HashMap<>();

    private boolean mHasInstalled = false;



    public void register(Module module) {
        if (module == null) {
            return;
        }
        mModules.put(module.getName(), module);
    }

    @Override
    public void unregister(Module module) {
        if (module == null) {
            return;
        }
        mModules.remove(module.getName());
    }

    public void unregister(String name) {
        mModules.remove(name);
    }

    @Override
    public void clear() {
        mModules.clear();
    }


    final void performInstalled(Application app) {
        if (mHasInstalled) {
            throw new UnsupportedOperationException("ModuleLoader had install modules yet");
        }
        mHasInstalled = true;
        List<Module> list = new ArrayList<>(mModules.values());
        Collections.sort(list, new Comparator<Module>() {
            @Override
            public int compare(Module o1, Module o2) {
                int diff = o2.getPriority() - o1.getPriority();
                if (diff > 0) {
                    return 1;
                } else if (diff < 0) {
                    return -1;
                } else {
                    return 0;
                }
            }
        });
        for (Module module : list) {
            module.install(app);
        }
    }



    final void performUninstalled(Application app) {
        for (Module module : mModules.values()) {
            module.uninstall(app);
        }
    }

    public Module getModule(String name) {
        return mModules.get(name);
    }

    @SuppressWarnings("unchecked")
    public <M extends Module> M getModule(String name, Class<M> cls) {
        Module m = getModule(name);
        if (m == null) {
            return null;
        }
        if (m.getClass() == cls) {
            return (M) m;
        }
        return null;
    }

    public int getModuleCount() {
        return mModules.size();
    }

}
