package android.dev.base.arch;

import android.annotation.SuppressLint;
import android.app.Application;
import android.content.ComponentCallbacks;
import android.dev.base.components.AppEnvironment;
import android.dev.base.components.MultiProcessLoader;
import android.os.Build;
import android.os.StrictMode;

/**
 * Created by wbs on 2017/12/7 0007.
 */

public final class AppContextLike {

    //make it single
    private final static AppContextLike IMPL = new AppContextLike();

    public static AppContextLike getDefault() {
        return IMPL;
    }

    private AppContextLike() {

    }

    //------------------------
    //Don't care
    @SuppressLint("StaticFieldLeak")
    private volatile static Application mDefault;

    //----------------------------------------

    private final ActivityStack mActivityStack = new ActivityStack();

    public void setApplication(Application application) {
        mDefault = application;
    }

    public void onCreate(Application application) {
        if (mDefault == null) {
            mDefault = application;
        }
        if (AppEnvironment.DEBUG) {
            StrictMode.VmPolicy.Builder builder = new StrictMode.VmPolicy.Builder()
                    .detectActivityLeaks()   //检测Activity泄漏
                    .detectLeakedSqlLiteObjects()//检测数据库泄漏
                    .detectLeakedClosableObjects();//检测Closeable对象泄漏
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                builder.detectLeakedRegistrationObjects();//检测注册对象的泄漏
            }

            StrictMode.setVmPolicy(builder.penaltyLog().build());
        }
        application.registerActivityLifecycleCallbacks(mActivityStack);
        ModuleLoader.getDefault().performInstalled(application);
        MultiProcessLoader.attach(application);
    }


    public void registerApplicationComponent(ComponentCallbacks callbacks){
        if(mDefault==null||callbacks==null){
            return;
        }
        mDefault.registerComponentCallbacks(callbacks);
    }
    public void unregisterApplicationComponent(ComponentCallbacks callbacks){
        if(mDefault==null||callbacks==null){
            return;
        }
        mDefault.unregisterComponentCallbacks(callbacks);
    }

    public void onTerminate(Application application) {
        try {
            exit(application, true);
        } catch (Exception ignore) {
        }
    }

    public final void appExit(boolean killProcess) {
        try {
            exit(mDefault, killProcess);
        } catch (Exception ignored) {
        }
    }

    private void exit(Application application, boolean killProcess) {
        mActivityStack.finishAllActivity();
        if (killProcess) {
            if (application != null) {
                application.unregisterActivityLifecycleCallbacks(mActivityStack);
                ModuleLoader.getDefault().performUninstalled(application);
            }
            ModuleLoader.getDefault().clear();
            android.os.Process.killProcess(android.os.Process.myPid());
            System.exit(0);
        }

    }

    public static <A extends Application> A getAppContext() {
        if (mDefault == null) {
            return null;
        }
        return (A) mDefault;
    }

    public ActivityStack getActivityStack() {
        return mActivityStack;
    }
}
