package android.dev.base.view;

import android.animation.ValueAnimator;
import android.annotation.TargetApi;
import android.content.Context;
import android.content.res.TypedArray;
import android.dev.base.R;
import android.dev.base.utils.ThemeUtils;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Rect;
import android.graphics.drawable.Animatable;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.View;

import java.util.ArrayList;

/**
 * Created by wbs on 2017/11/20 0020.
 */

public class DefaultLoadingView extends View implements AnimateView {

    private static final int MIN_SHOW_TIME = 500; // ms

    private static final int MIN_DELAY = 500; // ms

    private final int mMinWidth;

    private final int mMaxWidth;

    private final int mMinHeight;

    private final int mMaxHeight;

    private LoadingDrawable mDrawable;

    private boolean mShouldStartAnimationDrawable;

    private int mLoadingColor;

    private boolean mDismissed = false;

    private long mStartTime = -1;

    private ViewAction mAction;

    private final Runnable mDelayedHide = new Runnable() {

        @Override
        public void run() {
            mStartTime = -1;
            setVisibility(View.GONE);
            if (mAction != null) {
                mAction.onDidDisAppear();
            }
        }
    };

    private final Runnable mDelayedShow = new Runnable() {

        @Override
        public void run() {
            if (!mDismissed) {
                mStartTime = System.currentTimeMillis();
                setVisibility(View.VISIBLE);
                if (mAction != null) {
                    mAction.onDidAppear();
                }
            }
        }
    };

    public DefaultLoadingView(Context context) {
        this(context, null, 0);
    }

    public DefaultLoadingView(Context context, @Nullable AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public DefaultLoadingView(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        final TypedArray a = context.obtainStyledAttributes(
                attrs, R.styleable.DefaultLoadingView, defStyleAttr, defStyleAttr);
        mMinWidth = a.getDimensionPixelSize(R.styleable.DefaultLoadingView_minWidth, 24);
        mMaxWidth = a.getDimensionPixelSize(R.styleable.DefaultLoadingView_maxWidth, 48);
        mMinHeight = a.getDimensionPixelSize(R.styleable.DefaultLoadingView_minHeight, 24);
        mMaxHeight = a.getDimensionPixelSize(R.styleable.DefaultLoadingView_maxHeight, 48);
        mLoadingColor = parseLoadingColor(context, a);
        String loadingDrawable = a.getString(R.styleable.DefaultLoadingView_loadingDrawable);
        setLoadDrawable(loadingDrawable);
        if (mDrawable == null) {
            setLoadDrawable(new DefaultLoadingDrawable());
        }
        a.recycle();
    }

    public void setLoadDrawable(String name) {
        if (TextUtils.isEmpty(name)) {
            return;
        }
        try {
            Class<?> drawableClass = Class.forName(name);
            LoadingDrawable d = (LoadingDrawable) drawableClass.newInstance();
            setLoadDrawable(d);
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        } catch (InstantiationException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        }
    }

    public void setLoadDrawable(LoadingDrawable d) {
        if (mDrawable != d) {
            if (mDrawable != null) {
                mDrawable.setCallback(null);
                unscheduleDrawable(mDrawable);
            }

            mDrawable = d;
            //need to set indicator color again if you didn't specified when you update the indicator .
            setLoadingColor(mLoadingColor);
            if (d != null) {
                d.setCallback(this);
            }
            postInvalidate();
        }
    }

    public LoadingDrawable getLoadingDrawable() {
        return mDrawable;
    }

    public void setLoadingColor(int color) {
        this.mLoadingColor = color;
        if (mDrawable != null) {
            mDrawable.setColor(mLoadingColor);
        }
    }

    void startAnimation() {
        if (getVisibility() != VISIBLE) {
            return;
        }
        mShouldStartAnimationDrawable = true;
        postInvalidate();
    }

    void stopAnimation() {
        mDrawable.stop();
        mShouldStartAnimationDrawable = false;
        postInvalidate();
    }

    @Override
    protected boolean verifyDrawable(@NonNull Drawable who) {
        return who == mDrawable || super.verifyDrawable(who);
    }

    @Override
    public void setVisibility(int v) {
        if (getVisibility() != v) {
            super.setVisibility(v);
            if (v == GONE || v == INVISIBLE) {
                stopAnimation();
            } else {
                startAnimation();
            }
        }
    }

    @Override
    protected void onVisibilityChanged(View changedView, int visibility) {
        super.onVisibilityChanged(changedView, visibility);
        if (visibility == GONE || visibility == INVISIBLE) {
            stopAnimation();
        } else {
            startAnimation();
        }
    }

    @Override
    public void invalidateDrawable(Drawable dr) {
        if (verifyDrawable(dr)) {
            final Rect dirty = dr.getBounds();
            final int scrollX = getScrollX() + getPaddingLeft();
            final int scrollY = getScrollY() + getPaddingTop();

            invalidate(dirty.left + scrollX, dirty.top + scrollY,
                    dirty.right + scrollX, dirty.bottom + scrollY);
        } else {
            super.invalidateDrawable(dr);
        }
    }

    @Override
    protected void onSizeChanged(int w, int h, int oldw, int oldh) {
        updateDrawableBounds(w, h);
    }

    private void updateDrawableBounds(int w, int h) {
        // onDraw will translate the canvas so we draw starting at 0,0.
        // Subtract out padding for the purposes of the calculations below.
        w -= getPaddingRight() + getPaddingLeft();
        h -= getPaddingTop() + getPaddingBottom();

        int right = w;
        int bottom = h;
        int top = 0;
        int left = 0;

        if (mDrawable != null) {
            // Maintain aspect ratio. Certain kinds of animated drawables
            // get very confused otherwise.
            final int intrinsicWidth = mDrawable.getIntrinsicWidth();
            final int intrinsicHeight = mDrawable.getIntrinsicHeight();
            final float intrinsicAspect = (float) intrinsicWidth / intrinsicHeight;
            final float boundAspect = (float) w / h;
            if (intrinsicAspect != boundAspect) {
                if (boundAspect > intrinsicAspect) {
                    // New width is larger. Make it smaller to match height.
                    final int width = (int) (h * intrinsicAspect);
                    left = (w - width) / 2;
                    right = left + width;
                } else {
                    // New height is larger. Make it smaller to match width.
                    final int height = (int) (w * (1 / intrinsicAspect));
                    top = (h - height) / 2;
                    bottom = top + height;
                }
            }
            mDrawable.setBounds(left, top, right, bottom);
        }
    }

    @Override
    protected synchronized void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        drawTrack(canvas);
    }

    void drawTrack(Canvas canvas) {
        final Drawable d = mDrawable;
        if (d != null) {
            // Translate canvas so a indeterminate circular progress bar with padding
            // rotates properly in its animation
            final int saveCount = canvas.save();

            canvas.translate(getPaddingLeft(), getPaddingTop());

            d.draw(canvas);
            canvas.restoreToCount(saveCount);

            if (mShouldStartAnimationDrawable) {
                ((Animatable) d).start();
                mShouldStartAnimationDrawable = false;
            }
        }
    }

    @Override
    protected synchronized void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        int dw = 0;
        int dh = 0;

        final Drawable d = mDrawable;
        if (d != null) {
            dw = Math.max(mMinWidth, Math.min(mMaxWidth, d.getIntrinsicWidth()));
            dh = Math.max(mMinHeight, Math.min(mMaxHeight, d.getIntrinsicHeight()));
        }

        updateDrawableState();

        dw += getPaddingLeft() + getPaddingRight();
        dh += getPaddingTop() + getPaddingBottom();

        final int measuredWidth = resolveSizeAndState(dw, widthMeasureSpec, 0);
        final int measuredHeight = resolveSizeAndState(dh, heightMeasureSpec, 0);
        setMeasuredDimension(measuredWidth, measuredHeight);
    }

    @Override
    protected void drawableStateChanged() {
        super.drawableStateChanged();
        updateDrawableState();
    }

    private void updateDrawableState() {
        final int[] state = getDrawableState();
        if (mDrawable != null && mDrawable.isStateful()) {
            mDrawable.setState(state);
        }
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    @Override
    public void drawableHotspotChanged(float x, float y) {
        super.drawableHotspotChanged(x, y);

        if (mDrawable != null) {
            mDrawable.setHotspot(x, y);
        }
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        startAnimation();
        removeCallbacks();
    }

    @Override
    protected void onDetachedFromWindow() {
        stopAnimation();
        // This should come after stopAnimation(), otherwise an invalidate message remains in the
        // queue, which can prevent the entire view hierarchy from being GC'ed during a rotation
        super.onDetachedFromWindow();
        removeCallbacks();
    }

    private void removeCallbacks() {
        removeCallbacks(mDelayedHide);
        removeCallbacks(mDelayedShow);
    }

    private int parseLoadingColor(Context context, TypedArray a) {
        int loadingColor = a.getColor(R.styleable.DefaultLoadingView_loadingColor, -1);
        if (loadingColor == -1) {
            loadingColor = ThemeUtils.getThemeColorAttribute(context, R.attr.LoadingColor, -1);
        }
        if (loadingColor == -1) {
            loadingColor = ThemeUtils.getThemeColorAttribute(context, R.attr.colorPrimary, Color.WHITE);
        }
        return loadingColor;
    }

    @Override
    public void show() {
        // Reset the start time.
        mStartTime = -1;
        mDismissed = false;
        removeCallbacks(mDelayedHide);
        if (mAction != null) {
            mAction.onAppear();
        }
        removeCallbacks(mDelayedShow);
        postDelayed(mDelayedShow, MIN_DELAY);

    }

    @Override
    public void dismiss() {
        mDismissed = true;
        removeCallbacks(mDelayedShow);
        if (mAction != null) {
            mAction.onDisAppear();
        }
        long diff = System.currentTimeMillis() - mStartTime;
        if (diff >= MIN_SHOW_TIME || mStartTime == -1) {
            // The progress spinner has been shown long enough
            // OR was not shown yet. If it wasn't shown yet,
            // it will just never be shown.
            setVisibility(View.GONE);
            if (mAction != null) {
                mAction.onDidDisAppear();
            }
        } else {
            // The progress spinner is shown, but not long enough,
            // so put a delayed message in to hide it when its been
            // shown long enough.
            removeCallbacks(mDelayedHide);
            postDelayed(mDelayedHide, MIN_SHOW_TIME - diff);
        }
    }

    @Override
    public boolean isShowing() {
        return mDrawable != null && mDrawable.isRunning();
    }

    @Override
    public void registerViewAction(ViewAction action) {
        this.mAction = action;
    }

    private static class DefaultLoadingDrawable extends LoadingDrawable {

        private static final float SCALE = 1.0f;

        //scale x ,y
        private float[] scaleFloats = new float[]{SCALE,
                SCALE,
                SCALE,
                SCALE,
                SCALE};

        @Override
        public void draw(Canvas canvas, Paint paint) {
            float circleSpacing = 4;
            float radius = (Math.min(getWidth(), getHeight()) - circleSpacing * 2) / 12;
            float x = (getWidth() - (radius * 2 * 4 + 3 * circleSpacing)) / 2;
            float y = getHeight() / 2;
            final  float dis = radius*2+circleSpacing;
            for (int i = 0; i < 4; i++) {
                canvas.save();
                float translateX = x + radius+dis*i;
                canvas.translate(translateX, y);
                canvas.scale(scaleFloats[i], scaleFloats[i]);
                canvas.drawCircle(0, 0, radius, paint);
                canvas.restore();
            }
        }

        @Override
        public ArrayList<ValueAnimator> onCreateAnimators() {

            ArrayList<ValueAnimator> animators = new ArrayList<>();
            int[] delays = new int[]{120, 240, 360, 480};
            for (int i = 0; i < 4; i++) {
                final int index = i;

                ValueAnimator scaleAnim = ValueAnimator.ofFloat(1, 0.3f, 1);

                scaleAnim.setDuration(750);
                scaleAnim.setRepeatCount(-1);
                scaleAnim.setStartDelay(delays[i]);

                addUpdateListener(scaleAnim, new ValueAnimator.AnimatorUpdateListener() {
                    @Override
                    public void onAnimationUpdate(ValueAnimator animation) {
                        scaleFloats[index] = (float) animation.getAnimatedValue();
                        postInvalidate();

                    }
                });
                animators.add(scaleAnim);
            }
            return animators;
        }
    }
}
